import base64

from rest_framework import serializers

from cars.django.serializers import BaseSerializer
from cars.users.core.user_profile_updater import RecognizedDocument, RecognizedDocumentField


class ChatActionDataSerializer(BaseSerializer):
    pass


class PhotoChatActionDataSerializer(ChatActionDataSerializer):

    content = serializers.CharField()

    def validate_content(self, value):
        try:
            content = base64.b64decode(value)
        except ValueError:
            raise serializers.ValidationError('invalid_base64')
        return content


class RecognizedDocumentFieldSerializer(BaseSerializer):
    value = serializers.CharField()
    confidence = serializers.FloatField()

    def to_internal_value(self, data):
        return RecognizedDocumentField(
            value=data['value'],
            confidence=data['confidence'],
        )


class RecognizedDocumentFieldsSerializer(BaseSerializer):

    def to_internal_value(self, data):
        return {
            field_name: RecognizedDocumentFieldSerializer().to_internal_value(data=field_value)
            for field_name, field_value in data.items()
        }


class RecognizedDocumentSerializer(BaseSerializer):
    scanner = serializers.CharField()
    type = serializers.CharField()
    fields = RecognizedDocumentFieldsSerializer()

    def to_internal_value(self, data):
        return RecognizedDocument(
            scanner=data['scanner'],
            type_=data['type'],
            fields=self.fields['fields'].to_internal_value(data=data['fields']),
        )


class RecognizedDocumentChatActionSerializer(PhotoChatActionDataSerializer):
    scanners = serializers.ListField(child=RecognizedDocumentSerializer())


class CreditCardChatActionDataSerializer(ChatActionDataSerializer):

    class PanSerializer(BaseSerializer):
        prefix = serializers.CharField(min_length=6, max_length=6)
        suffix = serializers.CharField(min_length=4, max_length=4)

    pan = PanSerializer()


class DriverLicenseBackChatActionDataSerializer(RecognizedDocumentChatActionSerializer):
    pass

class DriverLicenseFrontChatActionDataSerializer(RecognizedDocumentChatActionSerializer):
    pass

class DriverLicenseChatActionDataSerializer(ChatActionDataSerializer):
    back = DriverLicenseBackChatActionDataSerializer()
    front = DriverLicenseFrontChatActionDataSerializer()


class PassportBiographicalChatActionDataSerializer(RecognizedDocumentChatActionSerializer):
    pass

class PassportRegistrationChatActionDataSerializer(RecognizedDocumentChatActionSerializer):
    pass

class PassportChatActionDataSerializer(ChatActionDataSerializer):
    biographical = PassportBiographicalChatActionDataSerializer()
    registration = PassportRegistrationChatActionDataSerializer()


class PassportSelfieChatActionDataSerializer(ChatActionDataSerializer):
    face = PhotoChatActionDataSerializer(allow_null=True)
    selfie = PhotoChatActionDataSerializer()

    def to_internal_value(self, data):
        if 'content' in data:
            # Support old format with a sole selfie.
            data = {
                'face': None,
                'selfie': data,
            }
        return super().to_internal_value(data)
