import logging
import time

from django.db import transaction

import cars.settings
from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView
from cars.users.models.registration_state import RegistrationState
from cars.users.core.user_profile_updater import UserHistoryManager
from cars.core.saas_drive_admin import SaasDriveAdminClient


LOGGER = logging.getLogger(__name__)


class RegistrationAPIView(CarsharingAPIView):
    permission_classes = [IsAuthenticated]

    def initial(self, request, *args, **kwargs):
        result = super().initial(request, *args, **kwargs)
        self._ensure_registration_state(request.user)
        try:
            self._update_registration_geo(request)
        except Exception:
            # this should never happen, but under no circumstance we
            # can leave a chance for registration break
            LOGGER.exception('unable to set user geo!')

        return result

    def _ensure_registration_state(self, user):
        registration_state = user.get_registration_state()
        if registration_state is None:
            RegistrationState.objects.create(user=user)

    def _update_registration_geo(self, request):
        """
        Reasoning is in https://st.yandex-team.ru/DRIVEBACK-494
        """
        user = request.user

        lat, lon = request.META.get('HTTP_LAT', 0.0), request.META.get('HTTP_LON', 0.0)

        try:
            lat, lon = float(lat), float(lon)
        except Exception:
            LOGGER.exception(
                'unable to set user geo as latitude and longitude have incorrect values: {} and {}'
                .format(lat, lon)
            )
            lat, lon = 0.0, 0.0

        LOGGER.info('geo {} : {} {}'.format(str(user.id), str(lat), str(lon)))

        if user.registration_geo is None:
            zones = cars.settings.REGISTRATION_GEOZONES
            for zone in zones:
                is_lat_matching = zone['min_lat'] <= lat <= zone['max_lat']
                is_lon_matching = zone['min_lon'] <= lon <= zone['max_lon']
                is_geo_matching = is_lat_matching and is_lon_matching
                if is_geo_matching:
                    user.registration_geo = zone['name']
                    with transaction.atomic(savepoint=False):
                        history_manager = UserHistoryManager()
                        user.save()
                        history_manager.update_entry(user)
