import logging
import time

from django.db import transaction
from rest_framework.status import HTTP_400_BAD_REQUEST
from rest_framework.response import Response

from cars.users.core.user_profile_updater import UserHistoryManager
from cars.users.models import User
from ..core.chat.actions import BaseChatAction
from ..core.chat.manager import ChatManager, ChatManagerSession
from ..serializers.chat import (
    ChatActionArgumentsSerializer, ChatStateSerializer, ChatViewArgumentsSerializer,
)
from .base import RegistrationAPIView


LOGGER = logging.getLogger(__name__)


class ChatActionView(RegistrationAPIView):

    chat_manager = ChatManager.from_settings()

    def post(self, request):
        serializer = ChatActionArgumentsSerializer(data=request.data)
        if not serializer.is_valid():
            data = {
                'errors': serializer.errors,
            }
            return Response(data, status=HTTP_400_BAD_REQUEST)

        try:
            chat_state = self.chat_manager.submit_chat_action(
                user=request.user,
                chat_action_id=serializer.validated_data['id'],
                data=serializer.validated_data['data'],
            )
        except (BaseChatAction.ValidationError, ChatManagerSession.ActionNotFound) as e:
            data = {
                'errors': str(e),
            }
            return Response(data, status=HTTP_400_BAD_REQUEST)

        data = ChatStateSerializer(chat_state).data

        return Response(data)


class ChatView(RegistrationAPIView):

    arguments_serializer_class = ChatViewArgumentsSerializer

    chat_manager = ChatManager.from_settings()

    def do_get(self, request):
#        chat_state = self.chat_manager.get_chat_state(
#            user=request.user,
#            since=request.arguments['since'],
#        )
#        data = ChatStateSerializer(chat_state).data
        data = {
            "action": None,
            "new_messages": [
                {
                    "id": 50759361,
                    "date": 1565457828.832122,
                    "chat_action_id": "block_other",
                    "source": "system",
                    "type": "text",
                    "content": {
                        "text": "Пожалуйста, обновите приложение Драйва"
                    }
                }
            ]
        }
        return Response(data)


class ResetChatView(RegistrationAPIView):

    chat_manager = ChatManager.from_settings()

    history_manager = UserHistoryManager()

    def post(self, request):
        user = request.user
        user.first_name = user.last_name = user.patronymic_name = ''
        user.phone = None
        user.status = User.Status.ONBOARDING.value

        registration_state = user.get_registration_state()

        session = self.chat_manager.make_session(user=user)
        session.reset_chat()
        user.updated_at = time.time()
        user.save()

        self.history_manager.update_entry(user)

        if registration_state is not None:
            registration_state.delete()

        return Response()
