import datetime
import logging
import os
import time
from collections import defaultdict

from django.db import transaction
from django.utils import timezone

import yt.wrapper as yt

import cars.settings
from cars.core.util import make_yt_client
from cars.registration_yang.core.datasync_helper import DataSyncHelper
from cars.registration_yang.core.enums import PhotoTypes
from cars.registration_yang.models import YangAssignment
from cars.users.models import User, UserDocumentPhoto, UserDocumentBackgroundVideo
from cars.users.core import UserProfileUpdater
from cars.users.core.datasync import DataSyncDocumentsClient


LOGGER = logging.getLogger(__name__)


class AssignmentSet:
    """
    Stores existing assignments for onboarding users so that we can do
    a duplicate check without a DB lookup.
    """

    def __init__(self, raw_assignments):
        self._assignments = set()
        for assignment in raw_assignments:
            key = self._get_assignment_key(assignment)
            self._assignments.add(key)
        LOGGER.info('Constructed an assignment set with %d entries', len(self._assignments))

    def __contains__(self, item):
        return self._get_assignment_key(item) in self._assignments

    def _get_assignment_key(self, assignment):
        return (
            assignment.license_back_id,
            assignment.license_front_id,
            assignment.passport_biographical_id,
            assignment.passport_registration_id,
            assignment.passport_selfie_id,
        )


class AssignmentManager:

    class AssignmentCreationError(Exception):
        pass

    document_types = {
        UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT.value: PhotoTypes.LICENSE_FRONT.value,
        UserDocumentPhoto.Type.DRIVER_LICENSE_BACK.value: PhotoTypes.LICENSE_BACK.value,
        UserDocumentPhoto.Type.DRIVER_LICENSE_SELFIE.value: PhotoTypes.LICENSE_SELFIE.value,
        UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL.value: PhotoTypes.PASSPORT_BIOGRAPHICAL.value,
        UserDocumentPhoto.Type.PASSPORT_REGISTRATION.value: PhotoTypes.PASSPORT_REGISTRATION.value,
        UserDocumentPhoto.Type.PASSPORT_SELFIE.value: PhotoTypes.PASSPORT_SELFIE.value,
        UserDocumentPhoto.Type.SELFIE.value: PhotoTypes.SELFIE.value,
    }

    def __init__(self, yt_client, assignment_tables_prefix, result_tables_prefix,
                 archive_results_prefix):
        self._client = yt_client
        self._assignment_tables_prefix = assignment_tables_prefix
        self._result_tables_prefix = result_tables_prefix
        self._archive_results_prefix = archive_results_prefix
        self._datasync = DataSyncHelper(self)

    @classmethod
    def from_settings(cls):
        assignment_tables_prefix = cars.settings.REGISTRATION_YANG['yt_tables']['assignments']
        result_tables_prefix = cars.settings.REGISTRATION_YANG['yt_tables']['results']
        archive_results_prefix = cars.settings.REGISTRATION_YANG['yt_tables']['results_archive']
        return cls(
            yt_client=make_yt_client('data'),
            assignment_tables_prefix=assignment_tables_prefix,
            result_tables_prefix=result_tables_prefix,
            archive_results_prefix=archive_results_prefix,
        )

    def get_photo_by_type(self, assignment, photo_type):
        if photo_type == PhotoTypes.LICENSE_FRONT.value:
            return assignment.license_front
        elif photo_type == PhotoTypes.LICENSE_BACK.value:
            return assignment.license_back
        elif photo_type == PhotoTypes.LICENSE_SELFIE.value:
            return assignment.license_selfie
        elif photo_type == PhotoTypes.PASSPORT_BIOGRAPHICAL.value:
            return assignment.passport_biographical
        elif photo_type == PhotoTypes.PASSPORT_REGISTRATION.value:
            return assignment.passport_registration
        elif photo_type == PhotoTypes.PASSPORT_SELFIE.value:
            return assignment.passport_selfie
        elif photo_type == PhotoTypes.SELFIE.value:
            return assignment.selfie
        else:
            raise KeyError('Incorrect photo type!')

    def get_latest_completed_assignment(self, uid):
        return (
            YangAssignment.objects
            .filter(license_back__user__uid=uid)
            .exclude(processed_at=None)
            .order_by('-created_at')
            .first()
        )

    def flush_assignment_list(self, assignments_lst, spb_assignments):
        # Save assignments into DB
        LOGGER.info('Flushing pool')
        with transaction.atomic():
            # Save the assignments first
            for assignment in assignments_lst:
                secret_id = str(assignment.id)
                LOGGER.info('Saving into base the assignment with id: %s', secret_id)
                assignment.save()

            # Prepare YT-format table
            secret_ids_yt_format = []
            for assignment in assignments_lst:
                secret_id = str(assignment.id)
                secret_ids_yt_format.append(
                    {
                        'secret_id': secret_id,
                    }
                )
                LOGGER.info('Adding to YT list assignment with id: %s', secret_id)

            # Save yt table
            table_path = self._assignment_tables_prefix
            if spb_assignments:
                table_path += '_new_cities'

            path = os.path.join(
                table_path,
                timezone.now().isoformat()
            )
            self._client.write_table(path, secret_ids_yt_format)

    def recreate_yang_assignment_for_users(self, user_ids, check_is_duplicate=True):
        assignments = []

        with transaction.atomic():
            for user_id in user_ids:
                assignment = self.recreate_yang_assignment_for_user(
                    user_id=user_id,
                    check_is_duplicate=check_is_duplicate,
                    flush=False,
                )
                assignments.append(assignment)

            self.flush_assignment_list(assignments, False)

        return assignments

    def recreate_yang_assignment_for_user(self, user_id, check_is_duplicate=True, flush=True):
        if check_is_duplicate:
            active_for_user = (
                YangAssignment.objects
                .filter(
                    license_back__user_id=user_id,
                    ingested_at__isnull=True,
                )
                .count()
            )
            if active_for_user > 0:
                raise self.AssignmentCreationError(
                    'There is already an active assignment for this user',
                )

        user_photos = (
            UserDocumentPhoto.objects
            .filter(user_id=user_id)
            .order_by('-submitted_at')
        )
        user_photos_lst = [p for p in user_photos]

        passport_biographical = self._get_last_photo_of_type(
            user_photos_lst=user_photos_lst,
            type_=UserDocumentPhoto.Type.PASSPORT_BIOGRAPHICAL.value,
        )
        passport_registration = self._get_last_photo_of_type(
            user_photos_lst=user_photos_lst,
            type_=UserDocumentPhoto.Type.PASSPORT_REGISTRATION.value,
        )
        passport_selfie = self._get_last_photo_of_type(
            user_photos_lst=user_photos_lst,
            type_=UserDocumentPhoto.Type.PASSPORT_SELFIE.value,
        )
        license_front = self._get_last_photo_of_type(
            user_photos_lst=user_photos_lst,
            type_=UserDocumentPhoto.Type.DRIVER_LICENSE_FRONT.value,
        )
        license_back = self._get_last_photo_of_type(
            user_photos_lst=user_photos_lst,
            type_=UserDocumentPhoto.Type.DRIVER_LICENSE_BACK.value,
        )

        all_photos_present = (
            passport_biographical is not None and
            passport_registration is not None and
            passport_selfie is not None and
            license_front is not None and
            license_back is not None
        )

        if not all_photos_present:
            raise self.AssignmentCreationError('Not all types of photos are present for this user')

        new_assignment = YangAssignment(
            created_at=timezone.now(),
            passport_biographical=passport_biographical,
            passport_registration=passport_registration,
            passport_selfie=passport_selfie,
            license_back=license_back,
            license_front=license_front,
        )

        if flush:
            self.flush_assignment_list([new_assignment], False)

        return new_assignment

    def create_new_assignments(self, fake=False, not_count_newest=False):
        """
        Will be moved to new backend in 1-2 weeks. Bad code.

        Need to take action now, in order not to make verification process for Moscow users slower.
        """
        total_new_assignments = 0
        for spb_presence in (False, True):
            user_candidates = self._get_candidate_users(spb_presence)
            existing_assignments = self._get_active_assignments()

            assignments_lst = []
            already_created_pool = False

            for user_uid, photos in user_candidates.items():
                if not isinstance(user_uid, int):
                    LOGGER.info('Processing fetched user UID: %s', str(user_uid))

                if not self._forms_valid_assignment(photos):
                    continue

                photos_assignment_format = self._to_assignment_format(photos)
                new_assignment = self._form_assignment(photos_assignment_format)

                if new_assignment not in existing_assignments:
                    most_recent_submission = max(
                        [
                            new_assignment.license_back.submitted_at,
                            new_assignment.license_front.submitted_at,
                            new_assignment.passport_biographical.submitted_at,
                            new_assignment.passport_registration.submitted_at,
                            new_assignment.passport_selfie.submitted_at,
                        ]
                    )
                    if not_count_newest:
                        if most_recent_submission + datetime.timedelta(minutes=15) < timezone.now():
                            total_new_assignments += 1
                    else:
                        total_new_assignments += 1

                    if fake:
                        continue
                    LOGGER.info('Seeing new assignment for UID: %s', str(user_uid))
                    forms_valid_assignment = self._forms_valid_assignment(photos, verbose=True)
                    LOGGER.info('Forms valid assignment: %s', str(forms_valid_assignment))
                    assignments_lst.append(new_assignment)
                    if len(assignments_lst) == cars.settings.REGISTRATION_YANG['creator']['batch_size']:
                        self.flush_assignment_list(assignments_lst, spb_presence)
                        assignments_lst = []
                        already_created_pool = True

            if assignments_lst and not already_created_pool and not fake:
                last_published_task = YangAssignment.objects.order_by('-created_at').first()
                if last_published_task is not None:
                    time_passed = timezone.now() - last_published_task.created_at
                    minutes = cars.settings.REGISTRATION_YANG['creator']['wait_minutes_to_create_pool']
                    is_time_to_create_pool = time_passed >= datetime.timedelta(minutes=minutes)
                    if is_time_to_create_pool:
                        self.flush_assignment_list(assignments_lst, spb_presence)
                elif not fake:
                    # If it is None, we should post assignments
                    self.flush_assignment_list(assignments_lst, spb_presence)

        return total_new_assignments

    def get_assignment_by_id(self, secret_id):
        assignment = (
            YangAssignment.objects
            .filter(id=secret_id).first()
        )
        return assignment

    def get_document_photo(self, secret_id, photo_type):
        assignment = self.get_assignment_by_id(secret_id)

        if not assignment:
            LOGGER.info('Assignment lookup failed, secret id: %s', secret_id)
            return None

        return self.get_photo_by_type(assignment, photo_type)

    def get_document_background_video(self, secret_id, photo_type):
        photo = self.get_document_photo(secret_id, photo_type)
        background_video = (
            UserDocumentBackgroundVideo.objects
            .filter(photo__id=photo.id).first()
        )
        return background_video

    def get_verification_status(self, secret_id, photo_type):
        photo = self.get_document_photo(secret_id, photo_type)
        return photo.verification_status

    def get_uid(self, secret_id):
        assignment = self.get_assignment_by_id(secret_id)
        return assignment.license_back.user.uid

    def update_user_from_table_row(self, row):
        secret_id = row['secretId']
        LOGGER.info('Processing for secretId: %s', secret_id)

        assignment = self.get_assignment_by_id(secret_id)
        if not assignment:
            LOGGER.error('The assignment with SecretID=%s does not exist', secret_id)
            return

        user = assignment.passport_biographical.user
        LOGGER.info('User UID: %s', str(user.uid))
        LOGGER.info('Photo statuses: %s', str(row['result']))

        statuses = row['result']
        if self._statuses_are_ok(statuses):
            is_final_assignment = True
            LOGGER.info('Statuses were considered as OK.')
        else:
            is_final_assignment = False

        copy_was_successful = False
        license_copied = False
        passport_copied = False
        for assignment_id in row['assignmentIds']:
            LOGGER.info('Trying to copy the data for assignmentId=%s', assignment_id)

            user.passport_ds_revision = assignment_id
            user.driving_license_ds_revision = assignment_id
            user.save()

            for attempt in range(3):
                license_copied = license_copied or self._datasync.copy_license_to_verified(user.uid, assignment_id)
                passport_copied = passport_copied or self._datasync.copy_passport_to_verified(user.uid, assignment_id)
                if passport_copied and license_copied:
                    break
                else:
                    LOGGER.error(
                        'unable to copy the persdata for user %s; key is %s; passport: %s ; license: %s',
                        str(user.uid),
                        str(assignment_id),
                        str(passport_copied),
                        str(license_copied)
                    )

                    actual_user_passport = self._datasync.client.get_passport_unverified(user.uid, user.passport_ds_revision)
                    actual_user_license = self._datasync.client.get_license_unverified(user.uid, user.driving_license_ds_revision)
                    if actual_user_passport:
                        passport_copied = True
                    if actual_user_license:
                        license_copied = True
                    if passport_copied and license_copied:
                        break

                    time.sleep(1)

            if passport_copied and license_copied:
                copy_was_successful = True
                break

        UserProfileUpdater(user=user).update_document_hashes()

        bad_copying = False
        creation_time = None
        if not copy_was_successful and is_final_assignment:
            # It is the final assignment for this user, but not all of the
            # data had been correctly copied. Thus we should create one more
            # assignment for this user in order to get the missing data
            LOGGER.error('No documents in datasync for user with UID: %d', user.uid)

            num_assignments_present = (
                YangAssignment.objects
                .filter(
                    license_back__user=user
                )
                .count()
            )
            limit = cars.settings.REGISTRATION_YANG['fetcher']['assignment_recreation_limit']
            LOGGER.info(
                'Currently there are %d assignments for this user.',
                num_assignments_present
            )
            if num_assignments_present < limit:
                self.recreate_yang_assignment_for_user(
                    user.id,
                    check_is_duplicate=False,
                )
                LOGGER.info('Task successfully recreated.')
            else:
                bad_copying = True
                creation_time = assignment.created_at

        LOGGER.info('Copying attempt to DataSync is done, changing photo statuses.')

        with transaction.atomic():
            assignment.processed_at = timezone.now()
            assignment.comments = row['comments']
            assignment.workers = row['workersIds']
            assignment.assignment_ids = row['assignmentIds']

            is_fraud = statuses['is_fraud']
            if is_fraud == 'NOT_FRAUD':
                assignment.is_fraud = YangAssignment.Status.NOT_FRAUD.value
            elif is_fraud == 'MAYBE_FRAUD':
                assignment.is_fraud = YangAssignment.Status.MAYBE_FRAUD.value
            elif is_fraud == 'DEFINITELY_FRAUD':
                assignment.is_fraud = YangAssignment.Status.DEFINITELY_FRAUD.value
            else:
                raise RuntimeError('unreachable: {}'.format(is_fraud))

            assignment.save()

            for photo_type in PhotoTypes:
                LOGGER.info('Updating photo of type: %s', photo_type.value)
                self.update_photo_status(
                    assignment=assignment,
                    statuses=statuses,
                    photo_type=photo_type.value,
                )

        if bad_copying:
            if creation_time + datetime.timedelta(hours=15) < timezone.now():
                # if 15 hours passed and nothing copied to DS, just let the table disappear
                pass
            else:
                # retry later. Exception will prevent that table from deletion
                raise RuntimeError('problem with datasync copying, waiting')

    def fetch_verification_results(self):
        result_tables = self._client.list(self._result_tables_prefix, absolute=True)
        had_failures = False
        for path in result_tables:
            rows = self._client.read_table(path, raw=False)
            all_successful = True
            with transaction.atomic():
                for row in rows:
                    try:
                        self.update_user_from_table_row(row)
                    except Exception:
                        # Don't stumble in case something weird happened for one user
                        # because we can always process others and at least not to
                        # waste time for them
                        LOGGER.exception('Error while updating the user from the table')
                        had_failures = True
                        all_successful = False
            if all_successful:
                self._move_to_archive(path)
        if had_failures:
            LOGGER.exception('Not all of the results were fetched from YT')

    def _move_to_archive(self, table_path):
        yt_table = os.path.join(self._archive_results_prefix, datetime.date.today().isoformat())
        self._client.write_table(
            yt.TablePath(yt_table, append=True),
            self._client.read_table(
                table_path,
                raw=False,
            ),
        )
        self._client.remove(table_path)

    def _form_assignment(self, photos):
        assignment = YangAssignment(
            created_at=timezone.now(),
            **photos
        )
        return assignment

    def _to_assignment_format(self, photos):
        """
        Convert photo fields from UserDocumentPhoto format (lb, lb, pb, pr, ps) to YangAssignment
        format (license_back, license_front, ...)
        """
        result = {}
        for type_, photo in photos.items():
            converted_name = self.document_types[type_]
            result[converted_name] = photo
        return result

    def _forms_valid_assignment(self, photos, verbose=False):
        if len(photos) < len(self.document_types):
            return False

        has_new_photo = False
        photo_statuses_ok = True

        for photo in photos.values():
            if verbose:
                LOGGER.info('User: %d', photo.user.uid)
                LOGGER.info('User status: %s', str(photo.user.status))
                LOGGER.info('PhotoID: %s', str(photo.id))
                LOGGER.info('Verification status: %s', str(photo.verification_status))
            if photo.verification_status is None:
                has_new_photo = True
                if verbose:
                    LOGGER.info(
                        'The verification status of this photo is None, so it is the new one',
                    )
            elif photo.verification_status != UserDocumentPhoto.VerificationStatus.OK.value:
                photo_statuses_ok = False
                if verbose:
                    LOGGER.info(
                        'The verification status differs from OK so it is a bad set of photos',
                    )

        if verbose:
            if has_new_photo:
                LOGGER.info('Has new photo')
            if photo_statuses_ok:
                LOGGER.info('Photo statuses OK')

        return has_new_photo and photo_statuses_ok

    def _get_active_assignments(self):
        assignments = (
            YangAssignment.objects
            .filter(processed_at__isnull=True)
        )
        return AssignmentSet(assignments)

    def _get_candidate_users(self, return_spb_users=False):
        """
        Get the information about onboarding users' documents.
        The result is dict where the user is mapped to their most recent docs.
        """
        users = defaultdict(dict)

        new_photos = (
            UserDocumentPhoto.objects
            .filter(verification_status__isnull=True)
            .select_related('document')
        )
        if return_spb_users:
            new_photos = new_photos.filter(user__registration_geo='spb')
        else:
            new_photos = new_photos.exclude(user__registration_geo='spb')

        users_awaiting_verification = set()
        for photo in new_photos:
            users_awaiting_verification.add(photo.user_id)

        new_documents = (
            UserDocumentPhoto.objects
            .filter(user_id__in=users_awaiting_verification)
            .prefetch_related('user')
            .order_by('submitted_at')
        )
        for photo in new_documents:
            users[photo.user.uid][photo.type] = photo

        LOGGER.info('There are %d candidate users.', len(users))

        return users

    def update_photo_status(self, assignment, statuses, photo_type):
        photo = self.get_photo_by_type(assignment, photo_type)

        # Set most recent verification status, even if we override some previous verification
        #   status of this particular photo object
        status = statuses[photo_type + '_status']

        try:
            photo.verification_status = UserDocumentPhoto.VerificationStatus[status].value  # get by name
        except ValueError:
            LOGGER.exception('invalid photo verification status: {}'.format(status))
            raise

        photo.verified_at = timezone.now()
        photo.save()

    def _statuses_are_ok(self, statuses):
        doc_statuses_are_ok = True
        for photo_type in self.document_types.values():
            status_key = photo_type + '_status'
            if statuses[status_key] != 'OK':
                doc_statuses_are_ok = False
                break
        return doc_statuses_are_ok

    def _get_last_photo_of_type(self, user_photos_lst, type_):
        for photo in user_photos_lst:
            if photo.type == type_:
                return photo
        return None
