import datetime

import dateutil.parser
from .enums import PhotoTypes


class FormatsConverter:

    # mapping from how the field named in datasync to how its' named in yang
    DATASYNC_TO_YANG_PASSPORT_BIOGRAPHICAL_MAPPING = [
        ('first_name', 'first_name'),
        ('last_name', 'last_name'),
        ('middle_name', 'middle_name'),
        ('birth_date', 'birth_date'),
        ('gender', 'gender'),
        ('citizenship', 'citizenship'),
        ('number', 'doc_value'),
        ('issue_date', 'issue_date'),
        ('birth_place', 'birth_place'),
        ('subdivision_code', 'subdivision_code'),
        ('expiration_date', 'expiration_date'),
        ('country', 'biographical_country'),
    ]
    DATASYNC_TO_YANG_PASSPORT_REGISTRATION_MAPPING = [
        ('region', 'registration_region'),
        ('area', 'registration_area'),
        ('locality', 'registration_locality'),
        ('street', 'registration_street'),
        ('house', 'registration_house'),
        ('housing', 'registration_housing'),
        ('apartment', 'registration_apartment'),
        ('registration_letter', 'registration_letter'),
        ('country', 'registration_country'),
        ('registration_expiration_date', 'registration_expiration_date'),
        ('type', 'registration_type')
    ]

    DATASYNC_TO_YANG_LICENSE_FRONT_MAPPING = [
        ('number', 'number_front'),
        ('issue_date', 'issue_date'),
        ('first_name', 'first_name'),
        ('last_name', 'last_name'),
        ('middle_name', 'middle_name'),
        ('birth_date', 'birth_date'),
        ('experience_from', 'experience_from'),
        ('categories', 'categories'),
        ('country', 'front_country')
    ]
    DATASYNC_TO_YANG_LICENSE_BACK_MAPPING = [
        ('number', 'number_back'),
        ('categories_b_valid_from_date', 'categories_b_valid_from_date'),
        ('categories_b_valid_to_date', 'categories_b_valid_to_date'),
        ('prev_licence_number', 'prev_licence_number'),
        ('prev_licence_issue_date', 'prev_licence_issue_date'),
        ('experience_from', 'experience_from'),
        ('country', 'back_country')
    ]

    # Date fields which need to be converted in DataSync format
    PASSPORT_DATE_FIELDS = [
        'issue_date',
        'birth_date',
        'registration_expiration_date',
        'expiration_date',
    ]

    LICENSE_DATE_FIELDS = [
        'issue_date',
        'birth_date',
        'prev_licence_issue_date',
        'experience_from',
        'categories_b_valid_from_date',
        'categories_b_valid_to_date',
    ]

    def from_yang_to_datasync(self, passport_biographical, passport_registration,
                              license_front, license_back):
        """
        Convert what Yang folks send us to what we can store in DataSync.
        """
        passport_biographical = passport_biographical.get('data', {})
        passport_registration = passport_registration.get('data', {})
        license_front = license_front.get('data', {})
        license_back = license_back.get('data', {})

        passport_data = {
            'doc_type': 'id',
        }
        license_data = {}

        self._add_converted_fields(
            result=passport_data,
            document=passport_biographical,
            mapping=self.DATASYNC_TO_YANG_PASSPORT_BIOGRAPHICAL_MAPPING,
        )

        self._add_converted_fields(
            result=passport_data,
            document=passport_registration,
            mapping=self.DATASYNC_TO_YANG_PASSPORT_REGISTRATION_MAPPING,
        )

        self._add_converted_fields(
            result=license_data,
            document=license_front,
            mapping=self.DATASYNC_TO_YANG_LICENSE_FRONT_MAPPING,
        )

        self._add_converted_fields(
            result=license_data,
            document=license_back,
            mapping=self.DATASYNC_TO_YANG_LICENSE_BACK_MAPPING,
        )

        self._prepare_date_fields(
            document=passport_data,
            date_fields=self.PASSPORT_DATE_FIELDS,
        )
        self._prepare_date_fields(
            document=license_data,
            date_fields=self.LICENSE_DATE_FIELDS
        )

        return {
            'passport_data': passport_data,
            'license_data': license_data,
        }

    def from_datasync_to_yang(self, passport_data, license_data):
        if passport_data is None:
            passport_data = {}
        if license_data is None:
            license_data = {}

        passport_biographical_data = {
            'data': {
                # required fields
                'first_name': passport_data.get('first_name'),
                'last_name': passport_data.get('last_name'),
                'middle_name': passport_data.get('middle_name'),
                'birth_date': self._transform_dsformat_to_date(passport_data.get('birth_date')),
                'expiration_date': self._transform_dsformat_to_date(passport_data.get('expiration_date')),
                'gender': passport_data.get('gender'),
                'citizenship': passport_data.get('citizenship'),
                # optional fields
                'number': passport_data.get('doc_value'),
                'issue_date': self._transform_dsformat_to_date(passport_data.get('issue_date')),
                'birth_place': passport_data.get('birth_place'),
                'subdivision_code': passport_data.get('subdivision_code'),
                'country': passport_data.get('biographical_country'),
            },
        }

        passport_registration_data = {
            'data': {
                'region': passport_data.get('registration_region'),
                'area': passport_data.get('registration_area'),
                'locality': passport_data.get('registration_locality'),
                'street': passport_data.get('registration_street'),
                'house': passport_data.get('registration_house'),
                'housing': passport_data.get('registration_housing'),
                'apartment': passport_data.get('registration_apartment'),
                'registration_letter': passport_data.get('registration_letter'),
                'country': passport_data.get('registration_country'),
                'registration_expiration_date': self._transform_dsformat_to_date(
                    passport_data.get('registration_expiration_date')
                ),
                'type': passport_data.get('registration_type'),
            },
        }

        passport_selfie_data = {}

        license_front_data = {
            'data': {
                'number': license_data.get('number_front'),
                'issue_date': self._transform_dsformat_to_date(license_data.get('issue_date')),
                'first_name': license_data.get('first_name'),
                'last_name': license_data.get('last_name'),
                'middle_name': license_data.get('middle_name'),
                'birth_date': self._transform_dsformat_to_date(license_data.get('birth_date')),
                'categories': license_data.get('categories'),
                'experience_from': self._transform_dsformat_to_date(
                    license_data.get('experience_from')
                ),
                'country': license_data.get('front_country'),
            },
        }

        license_back_data = {
            'data': {
                'number': license_data.get('number_back'),
                'categories_b_valid_from_date': self._transform_dsformat_to_date(
                    license_data.get('categories_b_valid_from_date')
                ),
                'categories_b_valid_to_date': self._transform_dsformat_to_date(
                    license_data.get('categories_b_valid_to_date')
                ),
                'prev_licence_number': license_data.get('prev_licence_number'),
                'prev_licence_issue_date': self._transform_dsformat_to_date(
                    license_data.get('prev_licence_issue_date')
                ),
                'experience_from': self._transform_dsformat_to_date(
                    license_data.get('experience_from')
                ),
                'country': license_data.get('back_country'),
            },
        }

        license_selfie_data = {}

        selfie_data = {}

        return {
            PhotoTypes.PASSPORT_BIOGRAPHICAL.value: passport_biographical_data,
            PhotoTypes.PASSPORT_REGISTRATION.value: passport_registration_data,
            PhotoTypes.PASSPORT_SELFIE.value: passport_selfie_data,
            PhotoTypes.LICENSE_FRONT.value: license_front_data,
            PhotoTypes.LICENSE_BACK.value: license_back_data,
            PhotoTypes.LICENSE_SELFIE.value: license_selfie_data,
            PhotoTypes.SELFIE.value: selfie_data,
        }

    def _transform_dsformat_to_date(self, time_isoformat):
        if time_isoformat is None:
            return None
        dt = dateutil.parser.parse(time_isoformat)
        if time_isoformat.endswith('21:00:00.000Z'):
            dt += datetime.timedelta(hours=3)
        return dt.date().isoformat()

    def _transform_date_to_dsformat(self, date_string):
        if date_string is None:
            return None
        return date_string + 'T00:00:00.000Z'

    def _add_converted_fields(self, result, document, mapping):
        for yang_field, datasync_field in mapping:
            if yang_field in document and document[yang_field]:
                result[datasync_field] = document[yang_field]

    def _prepare_date_fields(self, document, date_fields):
        for field in date_fields:
            if field in document:
                document[field] = self._transform_date_to_dsformat(document[field])
