import logging
import random

from requests.exceptions import RequestException

import cars.settings
from cars.users.core.datasync import DataSyncDocumentsClient
from cars.users.models import UserDocumentPhoto, User
from .converter import FormatsConverter
from .enums import Documents


DATASYNC_PASSPORT_ENDPOINT = '/v1/personality/profile/persdata/'

LOGGER = logging.getLogger(__name__)


class DataSyncHelper:

    client = DataSyncDocumentsClient.from_settings()

    converter = FormatsConverter()

    def __init__(self, assignment_manager):
        self.assignment_manager = assignment_manager

    def get_user_data(self, secret_id):
        if not secret_id:
            uid = cars.settings.REGISTRATION_YANG['preview_user_uid']
        else:
            uid = self.assignment_manager.get_uid(secret_id)

        prev_assignment = self.assignment_manager.get_latest_completed_assignment(uid)

        if not prev_assignment:
            datasync_ids = [
                'initial',
            ]
        else:
            datasync_ids = prev_assignment.assignment_ids

        passport_data = {}
        license_data = {}
        if not datasync_ids:
            datasync_ids = ['initial']

        random.shuffle(datasync_ids)

        for doc_id in datasync_ids:
            passport_data = self._get_data_from_datasync(uid, doc_id, Documents.PASSPORT)
            license_data = self._get_data_from_datasync(uid, doc_id, Documents.DRIVER_LICENSE)
            if passport_data and license_data:
                # There's a need for retry, if the data for some assignmentId was not
                # written to Yang for some reason (normally this shouldn't happen).
                break

        data = self.converter.from_datasync_to_yang(
            passport_data,
            license_data,
        )

        user = User.objects.get(uid=uid)
        if user.has_at_mark is not None and prev_assignment is not None:
            data['license_back']['data']['has_at_mark'] = user.has_at_mark
        else:
            data['license_back']['data']['has_at_mark'] = None

        for doc_type in data:
            if prev_assignment is None:
                status = None
            else:
                photo = self.assignment_manager.get_photo_by_type(prev_assignment, doc_type)
                status = photo.verification_status
            data[doc_type]['is_verified'] = status == UserDocumentPhoto.VerificationStatus.OK.value

        return data

    def update_user(self, uid, assignment_id, passport_biographical, passport_registration,
                    license_front, license_back):

        converted = self.converter.from_yang_to_datasync(
            passport_biographical,
            passport_registration,
            license_front,
            license_back,
        )

        passport_data = converted['passport_data']
        license_data = converted['license_data']

        if passport_data:
            self.client.update_passport_unverified(uid, assignment_id, passport_data)

        if license_data:
            self.client.update_license_unverified(uid, assignment_id, license_data)

    def copy_license_to_verified(self, uid, assignment_id):
        try:
            self.client.copy_license_to_verified(uid, assignment_id)
        except Exception:
            LOGGER.exception('Unable to copy license to verified collection, '
                             'uid=%s, assignment_id=%s', str(uid), assignment_id)
            return False

        return True

    def copy_passport_to_verified(self, uid, assignment_id):
        try:
            self.client.copy_passport_to_verified(uid, assignment_id)
        except Exception:
            LOGGER.exception('Unable to copy passport to verified collection, '
                             'uid=%s, assignment_id=%s', str(uid), assignment_id)
            return False

        return True

    def _get_data_from_datasync(self, uid, key, doc_type):
        try:
            if doc_type == Documents.PASSPORT:
                data = self.client.get_passport_unverified(uid, key)
            elif doc_type == Documents.DRIVER_LICENSE:
                data = self.client.get_license_unverified(uid, key)
            else:
                raise KeyError('Unknown document type!')
        except RequestException as e:
            LOGGER.exception('Exception while getting user data')
            if e.response is None or e.response.status_code != 404:
                raise e
            else:
                data = {}
        return data
