import json
import logging

import furl

from kubiki.util import make_requests_session

import cars.settings


LOGGER = logging.getLogger(__name__)


class YangSecretIdVerifier:

    def __init__(self, yang_verifier_base_url, oauth_token, verify_requests):
        self._session = make_requests_session()
        self._base_furl = furl.furl(yang_verifier_base_url)
        self._oauth_token = oauth_token
        self._verify_requests = verify_requests

    @classmethod
    def from_settings(cls):
        settings = cars.settings.REGISTRATION_YANG['verifier']
        return cls(
            yang_verifier_base_url=settings['base_url'],
            oauth_token=settings['token'],
            verify_requests=settings['verify_requests'],
        )

    def is_ok(self, assignment_id, secret_id):
        response = self._session.get(
            self._get_verify_url(assignment_id),
            headers={
                'Authorization': 'OAuth ' + self._oauth_token,
            },
            verify=self._verify_requests,
        )

        LOGGER.info('Response status code: %d', response.status_code)

        if response.status_code in (403, 404):
            return False
        response.raise_for_status()

        LOGGER.info('Response JSON: %s', json.dumps(response.json(), ensure_ascii=False).encode('utf8'))

        data = response.json()

        LOGGER.info('Response Status: %s', data['status'])
        is_status_ok = data['status'] == 'ACTIVE'
        LOGGER.info('Is status OK: %s', str(is_status_ok))
        is_secret_id_ok = False

        for entry in data['tasks']:
            LOGGER.info('Entry JSON: %s', json.dumps(entry, ensure_ascii=False).encode('utf8'))
            secret = entry.get('input_values', {}).get('secret')
            LOGGER.info('Secret: %s', str(secret))
            if str(secret).lower() == str(secret_id).lower():
                is_secret_id_ok = True
                break

        if not is_status_ok:
            LOGGER.info('Status is not OK: %s', data['status'])

        if not is_secret_id_ok:
            LOGGER.info('SecretId is not OK: %s', secret_id)

        return is_status_ok and is_secret_id_ok

    def _get_verify_url(self, assignment_id):
        request_furl = self._base_furl.copy()
        request_furl.join(str(assignment_id))
        return request_furl.url


class YangSecretIdVerifierStub:

    @classmethod
    def from_settings(cls):
        return cls()

    def is_ok(self, assignment_id, secret_id):
        LOGGER.info('verifying assignment %s with secret %s', assignment_id, secret_id)
        return True
