import time
import datetime
import logging
from decimal import Decimal

from django.db.models import Max, Sum
from django.utils import timezone

import cars.settings
from cars.core.daemons import CarsharingDaemon
from cars.registration.models import RegistrationTaxiRide
from cars.registration_yang.core.assignment import AssignmentManager
from cars.registration_yang.models import YangAssignment
from cars.users.models import RegistrationState, User, UserDocumentPhoto
from cars.registration.core.registration_manager import RegistrationManager


LOGGER = logging.getLogger(__name__)


class EntranceMonitoringDaemon(CarsharingDaemon):

    tick_interval = '* * * * *'

    def get_distributed_lock_relative_path(self):
        return 'yang/locks/entrance_monitoring.lock'

    def get_solomon_sensor_prefix(self):
        return 'daemons.entrance_monitoring'

    def get_solomon_service(self):
        return 'yang'

    def _monitor_real_reg_waiting_times(self, window_size, sensor_name):
        recently_registered_users_assignments = (
            YangAssignment.objects
            .filter(
                finalized_at__gte=timezone.now() - window_size
            )
            .select_related(
                'license_back__user'
            )
        )
        user_ids = [u.license_back.user_id for u in recently_registered_users_assignments]
        user_photos = (
            UserDocumentPhoto.objects
            .filter(
                user_id__in=user_ids
            )
            .only(
                'user_id',
                'submitted_at',
            )
        )
        last_photo_submission = {}
        for photo in user_photos:
            user_id = photo.user_id
            if user_id not in last_photo_submission:
                last_photo_submission[user_id] = photo.submitted_at
            elif last_photo_submission[user_id] < photo.submitted_at:
                last_photo_submission[user_id] = photo.submitted_at

        waiting_times = []
        for ya in recently_registered_users_assignments:
            if ya.license_back.user.registered_at and ya.finalized_at > ya.license_back.user.registered_at + datetime.timedelta(hours=2):
                continue
            if ya.finalized_at < last_photo_submission[ya.license_back.user_id]:
                continue
            if ya.license_back.user_id not in last_photo_submission:
                LOGGER.error('no photos found for user %s', str(ya.license_back.user_id))
                continue
            waiting_times.append((ya.finalized_at - last_photo_submission[ya.license_back.user_id]).total_seconds() / 3600)

        waiting_times.sort()
        BUCKETS = [15, 25, 35, 50, 75, 85, 95]
        for bucket in BUCKETS:
            print(bucket, waiting_times[len(waiting_times) * bucket // 100])
            self.solomon.set_value(sensor_name + "." + str(bucket), waiting_times[len(waiting_times) * bucket // 100])

    def _do_tick(self):
        self._monitor_real_reg_waiting_times(datetime.timedelta(days=1), 'yang.waiting_times.true_median')
        self._monitor_real_reg_waiting_times(datetime.timedelta(days=7), 'yang.waiting_times.true_median.7d')
