import datetime
import logging
import time
from concurrent.futures import ThreadPoolExecutor

import requests
from django.utils import timezone

import cars.settings
from cars.core.daemons import CarsharingDaemon
from cars.core.solomon import SolomonHelper
from ..models import YangAssignment


LOGGER = logging.getLogger(__name__)


class FixVideoDaemon(CarsharingDaemon):
    tick_interval = '* * * * * */15'

    solomon_helper = SolomonHelper('yang', 'video_fix')

    def get_distributed_lock_relative_path(self):
        return 'yang/locks/yang_video_fix.lock'

    def get_solomon_sensor_prefix(self):
        return 'yang.video_fix'

    def get_solomon_service(self):
        return 'yang'

    def _do_convert(self, request_url):
        for _ in range(4):
            success = self._do_convert_attempt(request_url)
            if not success:
                self.solomon_helper.increment_counter('error')
                time.sleep(1)
            else:
                self.solomon_helper.increment_counter('success')
                break
        else:
            LOGGER.error('video has not been converted: ' + request_url)
            self.solomon_helper.increment_counter('fail')

    def _do_convert_attempt(self, request_url):
        success = None

        try:
            r = requests.post(
                request_url,
                headers={
                    'Authorization': 'OAuth {}'.format(cars.settings.STAFF['token'])
                },
                timeout=20
            )
            r.raise_for_status()
            success = True
        except Exception:
            LOGGER.exception('unable to perform convertation request for ' + request_url)
            success = False

        return success

    def _do_tick(self):
        assignments = (
            YangAssignment.objects
            .filter(
                processed_at__isnull=True,
                created_at__gte=timezone.now() - datetime.timedelta(minutes=5)
            )
            .exclude(
                license_back__user_id='3cf2ab5b-dbba-4589-a4b1-5bcc7dd75413'  # QA should see the real picture
            )
        ).select_related('license_back__user')

        req_urls = []
        for assignment in assignments:
            user_id = str(assignment.license_back.user_id)

            photo_id_attr_names = [
                'license_back_id',
                'license_front_id',
                'license_selfie_id',
                'passport_biographical_id',
                'passport_registration_id',
                'passport_selfie_id',
            ]

            photo_ids = []
            for photo_id_attr_name in photo_id_attr_names:
                photo_id_attr_value = getattr(assignment, photo_id_attr_name)
                if photo_id_attr_value is not None:
                    photo_ids.append(str(photo_id_attr_value))
                    self.solomon_helper.increment_counter(photo_id_attr_name + '.queued')
                else:
                    LOGGER.warning('no video {} for user {} is available'.format(photo_id_attr_name, user_id))
                    self.solomon_helper.increment_counter(photo_id_attr_name + '.absent')

            LOGGER.info('queued for fix: ' + user_id)
            self.solomon_helper.increment_counter('user_queued')

            for photo_id in photo_ids:
                req_urls.append(
                    'https://carsharing.yandex-team.ru/api/admin/v1/users/{}/documents/null/photos/{}/background-video/convert/'
                    .format(user_id, photo_id)
                )

        with ThreadPoolExecutor(max_workers=24) as executor:
            list(executor.map(self._do_convert, req_urls))
