import datetime
import logging

from django.utils import timezone

import cars.settings
from cars.core.util import make_yt_client
from cars.core.daemons import CarsharingDaemon
from ..models import YangAssignment


LOGGER = logging.getLogger(__name__)


class MonitoringDaemon(CarsharingDaemon):

    tick_interval = '* * * * * */15'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.yt_client = make_yt_client('data')

    def get_distributed_lock_relative_path(self):
        return 'yang/locks/yang_monitoring.lock'

    def get_solomon_sensor_prefix(self):
        return 'yang.monitoring'

    def get_solomon_service(self):
        return 'yang'

    def _monitor_tasks_number(self, geo):
        """
        Monitor the number of tasks in the queue and the number of opened tasks.
        """
        active_tasks = self._get_all_active_tasks(geo).count()
        self.solomon.set_value(self._get_sensor_name('yang.active', geo), active_tasks)

    def _monitor_recently_opened_active_tasks(self, geo):
        """
        Monitor the number of tasks which were created during last 24 hours.
        """
        recently_opened_cnt = self._get_recently_opened_active_tasks(geo).count()
        self.solomon.set_value(self._get_sensor_name('yang.active.last24h', geo), recently_opened_cnt)

    def _monitor_recently_closed_tasks(self, geo):
        """
        Monitor the number of tasks which were completed during last 24 hours.
        """
        recently_closed_cnt = self._get_recently_closed_tasks(geo).count()
        self.solomon.set_value(self._get_sensor_name('yang.closed.last24h.count', geo), recently_closed_cnt)

    def _monitor_verification_times(self, geo):
        """
        Monitor the average time between task creation and task completion, in hours.
        """
        closed_tasks = self._get_recently_closed_tasks(geo)

        recently_closed_tasks = 0
        recently_closed_lifetime_sum = 0

        for task in closed_tasks:
            task_lifetime = (task.processed_at - task.created_at).total_seconds() / 3600.0
            recently_closed_tasks += 1
            recently_closed_lifetime_sum += task_lifetime

        if recently_closed_tasks == 0:
            recently_closed_tasks = 1

        self.solomon.set_value(
            self._get_sensor_name('yang.closed.last24h.avg_time', geo),
            recently_closed_lifetime_sum / recently_closed_tasks
        )

    def _monitor_recent_statuses(self, geo):
        """
        Monitor statuses of the recently closed tasks.
        """
        closed_tasks = self._get_recently_closed_tasks(geo)

        recently_fraudlent = 0
        recently_maybe_fraudlent = 0
        recently_not_fraudlent = 0

        for task in closed_tasks:
            if task.is_fraud == YangAssignment.Status.DEFINITELY_FRAUD.value:
                recently_fraudlent += 1
            elif task.is_fraud == YangAssignment.Status.MAYBE_FRAUD.value:
                recently_maybe_fraudlent += 1
            elif task.is_fraud == YangAssignment.Status.NOT_FRAUD.value:
                recently_not_fraudlent += 1

        self.solomon.set_value(self._get_sensor_name('yang.closed.last24h.fraud', geo), recently_fraudlent)
        self.solomon.set_value(self._get_sensor_name('yang.closed.last24h.maybe_fraud', geo), recently_maybe_fraudlent)
        self.solomon.set_value(self._get_sensor_name('yang.closed.last24h.not_fraud', geo), recently_not_fraudlent)

    def _monitor_task_creation_times(self, geo):
        """
        Set 1 for all times when the task was created.
        """
        opened_tasks = (
            self._get_recently_opened_active_tasks(geo)
            .filter(
                created_at__gte=timezone.now() - datetime.timedelta(minutes=15),
            )
        )

        for task in opened_tasks:
            self.solomon.set_value(self._get_sensor_name('yang.all_tasks.created_at', geo), 1, ts_datetime=task.created_at)

    def _monitor_recently_opened_tasks(self, geo):
        """
        Monitor the number of tasks which were created during last 24h, and
        are not necessarily active now.
        """
        opened_last_24h = self._get_all_recently_created_tasks(geo).count()

        self.solomon.set_value(self._get_sensor_name('yang.opened.last24h', geo), opened_last_24h)

    def _monitor_yt_tables_in_output_queue(self):
        """
        Monitor the number of tasks which were completed by Yang but not yet
        fetched by daemon.
        """
        yt_results_dir = cars.settings.REGISTRATION_YANG['yt_tables']['results']
        tables = self.yt_client.list(yt_results_dir)
        self.solomon.set_value('yang.verified_not_fetched.count', len(tables))

    def _get_recently_closed_tasks(self, geo):
        qs = (
            YangAssignment.objects
            .filter(
                processed_at__isnull=False,
                processed_at__gte=timezone.now() - datetime.timedelta(days=1),
            )
        )
        return self._filter_qs_by_geo(qs, geo)

    def _get_recently_opened_active_tasks(self, geo):
        return (
            self._get_all_active_tasks(geo)
            .filter(
                created_at__gte=timezone.now() - datetime.timedelta(days=1),
            )
        )

    def _get_all_active_tasks(self, geo):
        qs = YangAssignment.objects.filter(processed_at__isnull=True)
        return self._filter_qs_by_geo(qs, geo)

    def _get_all_recently_created_tasks(self, geo):
        qs = (
            YangAssignment.objects
            .filter(
                created_at__gte=timezone.now() - datetime.timedelta(days=1)
            )
        )
        return self._filter_qs_by_geo(qs, geo)

    # geo-related service stuff

    def _filter_qs_by_geo(self, qs, geo):
        if geo == 'all':
            return qs
        if geo == 'moscow':
            # if no geo is reported, we consider it moscow for now
            return qs.filter(license_back__user__registration_geo__in=['moscow', None])
        else:
            return qs.filter(license_back__user__registration_geo=geo)

    def _get_sensor_name(self, name, geo):
        if geo == 'all':
            return name
        else:
            return name + '.' + geo

    def _do_tick(self):
        for geo in ['moscow', 'spb', 'all']:
            self._monitor_tasks_number(geo)
            self._monitor_recently_opened_active_tasks(geo)
            self._monitor_recently_closed_tasks(geo)
            self._monitor_verification_times(geo)
            self._monitor_recent_statuses(geo)
            self._monitor_task_creation_times(geo)
            self._monitor_recently_opened_tasks(geo)
            self._monitor_yt_tables_in_output_queue()
