import enum
import uuid

from django.db import models
from django.contrib.postgres.fields import ArrayField

from cars.users.models.user_documents import UserDocumentPhoto


class YangAssignment(models.Model):

    class Status(enum.Enum):
        NOT_FRAUD = 'not_fraud'
        MAYBE_FRAUD = 'maybe_fraud'
        DEFINITELY_FRAUD = 'definitely_fraud'

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    passport_biographical = models.ForeignKey(
        UserDocumentPhoto,
        related_name='assignment_passport_biographical',
        on_delete=models.CASCADE,
    )
    passport_registration = models.ForeignKey(
        UserDocumentPhoto,
        related_name='assignment_passport_registration',
        on_delete=models.CASCADE,
    )
    passport_selfie = models.ForeignKey(
        UserDocumentPhoto,
        related_name='assignment_passport_selfie',
        on_delete=models.CASCADE,
    )
    license_front = models.ForeignKey(
        UserDocumentPhoto,
        related_name='assignment_license_front',
        on_delete=models.CASCADE,
    )
    license_back = models.ForeignKey(
        UserDocumentPhoto,
        related_name='assignment_license_back',
        on_delete=models.CASCADE,
    )
    license_selfie = models.ForeignKey(
        UserDocumentPhoto,
        related_name='assignment_license_selfie',
        on_delete=models.CASCADE,
    )
    selfie = models.ForeignKey(
        UserDocumentPhoto,
        related_name='selfie',
        on_delete=models.CASCADE,
    )

    is_fraud = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in Status],
        null=True,
    )

    comments = ArrayField(
        models.TextField(),
        null=True,
    )

    workers = ArrayField(
        models.CharField(
            max_length=32,
        ),
        null=True,
    )

    assignment_ids = ArrayField(
        models.CharField(
            max_length=64,
        ),
        null=True,
    )

    created_at = models.DateTimeField()
    processed_at = models.DateTimeField(null=True, blank=True)
    ingested_at = models.DateTimeField(null=True, blank=True)

    is_experimental = models.NullBooleanField()
    finalized_at = models.DateTimeField(null=True, blank=True)
    fraud_reasons = models.TextField(null=True, blank=True)
    history = models.TextField(null=True, blank=True)

    class Meta:
        db_table = 'yang_assignments'
        indexes = [
            models.Index(
                fields=['processed_at', 'ingested_at'],
                name='yang_assignments_ready_to_ingestion_idx',
            ),
            models.Index(
                fields=['created_at'],
                name='yang_assignments_created_at_idx',
            ),
        ]

    def __str__(self):
        return str(self.id)
