from abc import abstractmethod

from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST, HTTP_403_FORBIDDEN, HTTP_501_NOT_IMPLEMENTED

from cars.django.serializers import BaseSerializer

import cars.settings
from cars.core.mds.wrapper import MDSDocumentsWrapper
from cars.core.util import import_class
from cars.django.views import CarsharingAPIView
from ..core.assignment import AssignmentManager


class RegistrationYangView(CarsharingAPIView):

    serializer_class = BaseSerializer

    bucket = cars.settings.USERS['documents']['mds']['bucket_name']

    mds_client = MDSDocumentsWrapper.from_settings()

    assignment_manager = AssignmentManager.from_settings()

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self._verifier = self._make_verifier()

    def _make_verifier(self):
        verifier_class = import_class(cars.settings.REGISTRATION_YANG['verifier']['class'])
        verifier = verifier_class.from_settings()
        return verifier

    def get(self, request, *args, **kwargs):
        assignment_id = request.GET.get('assignmentId')
        if assignment_id == 'undefined':
            return self._get_preview()

        auth_error_response = self._authenticate(request, kwargs.get('secret_id'))
        if auth_error_response is not None:
            return auth_error_response
        return super().get(request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        auth_error_response = self._authenticate(request, kwargs.get('secret_id'))
        if auth_error_response is not None:
            return auth_error_response
        return super().post(request, *args, **kwargs)

    def _authenticate(self, request, secret_id):
        assignment_id = request.GET.get('assignmentId')

        if assignment_id == 'undefined':
            # Always allow preview with fake passport
            return None

        if assignment_id is None or secret_id is None:
            return Response(
                status=HTTP_400_BAD_REQUEST,
                data={
                    'error': 'No assignmentId specified',
                },
                content_type='application/json',
            )

        if not self._verifier.is_ok(assignment_id, secret_id):
            return Response(
                # balancer retries 5xx, so if we do a debug rollout on one DC,
                # users won't see that something doesn't work
                # 4xx are not retried by balancer policy, so experimental rollouts
                # with 4xx may harm verification process
                status=HTTP_501_NOT_IMPLEMENTED,
                data={
                    'error': 'secret_id/assignment_id pair mismatch',
                },
                content_type='application/json',
            )

    @abstractmethod
    def _get_preview(self):
        raise NotImplementedError
