import logging
from abc import abstractmethod

from rest_framework.response import Response
from rest_framework.status import HTTP_404_NOT_FOUND

import cars.settings
from cars.django.views import StreamingResponseViewMixin
from .base import RegistrationYangView
from ..core.enums import PhotoTypes


LOGGER = logging.getLogger(__name__)


class UserDocumentBackgroundVideoView(StreamingResponseViewMixin, RegistrationYangView):

    @abstractmethod
    def get_photo_type(self):
        raise NotImplementedError

    def do_get(self, _, secret_id):
        return self._get_video_response(secret_id)

    def _get_video_response(self, secret_id):
        user_document_background_video = self.assignment_manager.get_document_background_video(
            secret_id,
            self.get_photo_type(),
        )

        if user_document_background_video is None:
            return Response(status=HTTP_404_NOT_FOUND)

        mds_response = self.mds_client.get_user_document_background_video(
            user_document_background_video,
        )

        stream = mds_response['Body']
        content_type = user_document_background_video.mime_type
        content_length = mds_response['ContentLength']

        return self.get_response(
            stream=stream,
            content_type=content_type,
            content_length=content_length,
        )

    def _get_preview(self):
        return self._get_video_response(cars.settings.REGISTRATION_YANG['preview_user_uid'])


class DriverLicenseBackBackgroundVideoView(UserDocumentBackgroundVideoView):
    def get_photo_type(self):
        return PhotoTypes.LICENSE_BACK.value


class DriverLicenseFrontBackgroundVideoView(UserDocumentBackgroundVideoView):
    def get_photo_type(self):
        return PhotoTypes.LICENSE_FRONT.value


class DriverLicenseSelfieBackgroundVideoView(UserDocumentBackgroundVideoView):
    def get_photo_type(self):
        return PhotoTypes.LICENSE_SELFIE.value


class PassportBiographicalBackgroundVideoView(UserDocumentBackgroundVideoView):
    def get_photo_type(self):
        return PhotoTypes.PASSPORT_BIOGRAPHICAL.value


class PassportRegistrationBackgroundVideoView(UserDocumentBackgroundVideoView):
    def get_photo_type(self):
        return PhotoTypes.PASSPORT_REGISTRATION.value


class PassportSelfieBackgroundVideoView(UserDocumentBackgroundVideoView):
    def get_photo_type(self):
        return PhotoTypes.PASSPORT_SELFIE.value


class SelfieBackgroundVideoView(UserDocumentBackgroundVideoView):
    def get_photo_type(self):
        return PhotoTypes.SELFIE.value

