import logging

from rest_framework.response import Response
from rest_framework.status import HTTP_404_NOT_FOUND

import cars.settings
from cars.core.images import ImageProcessor, Resizer, Watermarker
from cars.django.renderers import JPEGRenderer
from .base import RegistrationYangView
from ..core.enums import PhotoTypes

LOGGER = logging.getLogger(__name__)


class UserDocumentPhotoView(RegistrationYangView):

    renderer_classes = [JPEGRenderer]

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        self._watermarker_props = cars.settings.REGISTRATION_YANG['document_images']['watermarker']

        self._resizer = Resizer(
            cars.settings.REGISTRATION_YANG['document_images']['greater_side_to_fit']
        )

    def get_type(self):
        raise NotImplementedError

    def do_get(self, request, secret_id):
        assignment_id = request.GET.get('assignmentId')
        return self._get_image_response(secret_id, assignment_id)

    def _get_image_response(self, secret_id, assignment_id):
        user_document_photo = self.assignment_manager.get_document_photo(secret_id, self.get_type())
        if user_document_photo is None:
            return Response(status=HTTP_404_NOT_FOUND)

        watermarker = Watermarker(
            base_watermark_size=self._watermarker_props['base_watermark_size'],
            watermark_visibility=self._watermarker_props['watermark_visibility'],
            font_size=self._watermarker_props['font_size'],
            watermark_first_line=assignment_id[:18],
            watermark_second_line=assignment_id[18:],
        )

        image_processor = ImageProcessor(
            watermarker=watermarker,
            resizer=self._resizer,
        )

        processed = self.mds_client.get_user_document_photo(
            user_document_photo=user_document_photo,
            image_processor=image_processor,
        )

        response = Response(processed, content_type='image/jpeg')
        return response

    def _get_preview(self):
        return self._get_image_response(
            cars.settings.REGISTRATION_YANG['preview_user_uid'],
            'undefined',
        )


class DriverLicenseBackPhotoView(UserDocumentPhotoView):
    def get_type(self):
        return PhotoTypes.LICENSE_BACK.value


class DriverLicenseFrontPhotoView(UserDocumentPhotoView):
    def get_type(self):
        return PhotoTypes.LICENSE_FRONT.value


class DriverLicenseSelfiePhotoView(UserDocumentPhotoView):
    def get_type(self):
        return PhotoTypes.LICENSE_SELFIE.value


class PassportBiographicalPhotoView(UserDocumentPhotoView):
    def get_type(self):
        return PhotoTypes.PASSPORT_BIOGRAPHICAL.value


class PassportRegistrationPhotoView(UserDocumentPhotoView):
    def get_type(self):
        return PhotoTypes.PASSPORT_REGISTRATION.value


class PassportSelfiePhotoView(UserDocumentPhotoView):
    def get_type(self):
        return PhotoTypes.PASSPORT_SELFIE.value


class SelfiePhotoView(UserDocumentPhotoView):
    def get_type(self):
        return PhotoTypes.SELFIE.value

