import logging
import time

import requests
from django.db import transaction
from django.utils import timezone
from requests.exceptions import RequestException
from rest_framework.response import Response

from cars.registration_yang.models import YangAssignment
from cars.users.core.user_profile_updater import UserHistoryManager
from cars.users.models import User
from ..core.assignment import AssignmentManager
from ..core.datasync_helper import DataSyncHelper
from ..core.enums import PhotoTypes
from .base import RegistrationYangView


LOGGER = logging.getLogger(__name__)


class DocumentsDataView(RegistrationYangView):

    assignment_manager = AssignmentManager.from_settings()
    ds_helper = DataSyncHelper(assignment_manager)
    history_manager = UserHistoryManager()

    def do_get(self, request, secret_id):
        assignment_id = request.GET.get('assignmentId')
#        try:
#            r = requests.get(
#                'https://stable.carsharing.yandex.net/support_api/yang/assignment/data?assignmentId={}&secretId={}'.format(assignment_id, secret_id)
#            )
#            r.raise_for_status()
#            LOGGER.info('Successful documents_data proxy request')
#            return Response(r.json())
#        except Exception:
#            LOGGER.exception('Failure in documents_data proxy request')
        return Response(self.ds_helper.get_user_data(secret_id))

    def do_post(self, request, secret_id):
        assignment_id = request.GET.get('assignmentId')
#        try:
#            r = requests.post(
#                'https://stable.carsharing.yandex.net/support_api/yang/assignment/data?assignmentId={}&secretId={}'.format(assignment_id, secret_id),
#                json=request.data
#            )
#            r.raise_for_status()
#            LOGGER.info('Successful POST documents_data proxy request ' + secret_id)
#            return Response()
#        except Exception:
#            LOGGER.exception('Failure in POST documents_data proxy request ' + secret_id)
        return self.do_post_straight(request, secret_id)

    def do_post_straight(self, request, secret_id):
        uid = self.assignment_manager.get_uid(secret_id)
        assignment_id = request.GET.get('assignmentId')

        passport_biographical = request.data.get(
            PhotoTypes.PASSPORT_BIOGRAPHICAL.value,
            {},
        ) or {}
        passport_registration = request.data.get(
            PhotoTypes.PASSPORT_REGISTRATION.value,
            {},
        ) or {}
        license_front = request.data.get(
            PhotoTypes.LICENSE_FRONT.value,
            {},
        ) or {}
        license_back = request.data.get(
            PhotoTypes.LICENSE_BACK.value,
            {},
        ) or {}

        try:
            self.ds_helper.update_user(
                uid=uid,
                assignment_id=assignment_id,
                passport_biographical=passport_biographical,
                passport_registration=passport_registration,
                license_front=license_front,
                license_back=license_back,
            )

            user = User.objects.get(uid=uid)
            if 'has_at_mark' in license_back.get('data', {}) and license_back['data']['has_at_mark'] is not None:
                user.has_at_mark = license_back['data']['has_at_mark']

            user.passport_ds_revision = assignment_id
            user.driving_license_ds_revision = assignment_id

            if passport_biographical.get('first_name') not in (None, ''):
                user.first_name = passport_biographical.get('first_name').strip().capitalize()
            elif license_front.get('first_name') not in (None, ''):
                user.first_name = license_front.get('first_name').strip().capitalize()

            if passport_biographical.get('last_name') not in (None, ''):
                user.last_name = passport_biographical.get('last_name').strip().capitalize()
            elif license_front.get('last_name') not in (None, ''):
                user.last_name = license_front.get('last_name').strip().capitalize()

            if passport_biographical.get('middle_name') not in (None, ''):
                user.patronymic_name = passport_biographical.get('middle_name').strip().capitalize()
            elif license_front.get('middle_name') not in (None, ''):
                user.patronymic_name = license_front.get('middle_name').strip().capitalize()

            user.updated_at = time.time()

            with transaction.atomic(savepoint=False):
                user.save()
                self.history_manager.update_entry(user)

        except RequestException as e:
            LOGGER.exception(
                'Saving to unverified DS failed, uid=%s, ya-cloud-req-id=%s',
                str(uid),
                str(
                    e.response.headers.get('Yandex-Cloud-Request-ID')
                    if e.response is not None
                    else None
                )
            )
            if e.response is not None and e.response.status_code < 500:
                return Response(status=e.response.status_code)
            else:
                raise e

        if 'verification_statuses' in request.data:
            try:
                statuses = request.data['verification_statuses']
                assignment = YangAssignment.objects.get(id=secret_id)

                with transaction.atomic(savepoint=False):
                    for photo_type in PhotoTypes:
                        LOGGER.info('Updating photo of type: %s', photo_type.value)
                        self.assignment_manager.update_photo_status(
                            assignment=assignment,
                            statuses=statuses,
                            photo_type=photo_type.value,
                        )
                    assignment.save()

                is_fraud = statuses['is_fraud']

                try:
                    assignment.is_fraud = YangAssignment.Status[is_fraud].value  # by name
                except ValueError:
                    LOGGER.exception('invalid fraud status: {}'.format(is_fraud))
                    raise

                assignment.processed_at = timezone.now()
                assignment.ingested_at = None
                assignment.assignment_ids = [assignment_id, ]
                if 'comment' in statuses and statuses['comment']:
                    assignment.comments = assignment.comments or []
                    assignment.comments.append(statuses['comment'])
                assignment.is_experimental = True
                assignment.save()

            except Exception:
                LOGGER.exception('prototype fast registration failed')

        return Response()

    def _get_preview(self):
        return Response(self.ds_helper.get_user_data(None))
