import itertools
import logging

from cars.core.saas_drive_admin import SaasDriveAdminClient

from ..core.audiotele import AudioteleProcessingHelper
from ..core.beeper import BeeperProcessingHelper
from ..core.chat2desk import Chat2DeskProcessingHelper
from ..core.internal_cc import IncomingTelephonyProcessingHelper, CCInternalAltayOutProcessingHelper
from ..core.push import OrdinaryPushHistoryHelper, TagNotificationsHistoryHelper
from ..core.appmetrica_push import AppMetricaPushHelper


LOGGER = logging.getLogger(__name__)


class AggregatedProcessingHelper(object):
    def __init__(self, *helpers):
        self._helpers = helpers
        self._saas_client = SaasDriveAdminClient.from_settings()

    @classmethod
    def from_settings(cls):
        incoming_telephony_helper = IncomingTelephonyProcessingHelper.from_settings()
        outgoing_telephony_helper = CCInternalAltayOutProcessingHelper.from_settings()
        audiotele_helper = AudioteleProcessingHelper.from_settings()
        beeper_helper = BeeperProcessingHelper.from_settings()
        chat2desk_helper = Chat2DeskProcessingHelper.from_settings()
        # ordinary_push_helper = OrdinaryPushHistoryHelper.from_settings()
        # tag_notifications_helper = TagNotificationsHistoryHelper.from_settings()
        # metric_push_helper = AppMetricaPushHelper.from_settings()

        return cls(
            incoming_telephony_helper,
            outgoing_telephony_helper,
            audiotele_helper,
            beeper_helper,
            chat2desk_helper
            # ordinary_push_helper,
            # tag_notifications_helper,
            # metric_push_helper
        )

    def get_entries(self, request=None, request_type=None, **kwargs):
        request_collections = []

        for helper in self._helpers:
            if helper.check_request_type(request_type):
                try:
                    entries = helper.get_requests(request=request, **kwargs)
                    if entries:
                        request_collections.append(entries)
                except Exception:
                    LOGGER.exception('Failed to fetch %s response',
                                     type(helper).__name__)

        # complexity is the same to heapq.merge
        response_entries = sorted(
            itertools.chain.from_iterable(request_collections),
            key=lambda x: x['time_enter'], reverse=True
        )

        # temporary
        has_access_to_queue_name = not self._saas_client.check_user_role(
            'team_adm_level3', request=request, require_all=False
        )
        if not has_access_to_queue_name:
            for entry in response_entries:
                entry['request']['queue'] = ''

        return response_entries
