import datetime as dt
import requests
import urllib

import siphash

from cars.callcenter.serializers.user import StaffUserEntrySerializer
from cars.core.tvm2 import TVM2ServiceTicket
from cars.request_aggregator.core.push.processing_helper import BasePushHistoryHelper
from cars.request_aggregator.serializers.common import AggregatedResponseEntry
import cars.settings
from cars.settings import APPMETRICA as settings
from cars.users.models.devices import UserDevices


METRICA_CGI_PARAMS = {
    'lang': 'ru',
    'request_domain': 'ru',
    'id': settings['APPLICATION_ID'],
    'group': 'day',
    'metrics': 'ym:pc:sentDevices,ym:pc:receivedDevices',
    'dimensions': 'ym:pc:campaignOrGroup',
    'accuracy': 1,
    'proposedAccuracy': 'true',
}
# TODO(kirr): AppMetrica become slower on requests with long date periods.
# So maybe it's better to import data day by day to our DB instead
# asking metrica.
DEFAULT_VIEW_PERIOD = dt.timedelta(days=30)
RESPONSE_TIMEOUT_SECONDS = 1


def sip(device_id):
    return siphash.SipHash_2_4(b'\x00'*16, bytes(device_id, encoding='ascii')).hash()


def datetime_to_date_str(date):
    return date.strftime('%Y-%m-%d')


def build_appmetrica_request(filters):
    params = METRICA_CGI_PARAMS.copy()
    params.update(filters)
    return '{}?{}'.format(
            urllib.parse.urljoin(settings['URL'], '/stat/v1/data/bytime'),
            urllib.parse.urlencode(params))


class AppMetricaPushHelper(BasePushHistoryHelper):
    def __init__(self, ticket, denied_operator_view_roles):
        super().__init__(denied_operator_view_roles)
        self._ticket = ticket

    def _get_headers(self):
        headers = {}
        if self._ticket:
            headers = {'X-Ya-Service-Ticket': self._ticket.content}
        return headers

    @classmethod
    def from_settings(cls):
        tvm_section = settings.get('tvm2')
        ticket = TVM2ServiceTicket(**tvm_section) if tvm_section else None
        denied_operator_view_roles = cars.settings.REQUEST_AGGREGATOR['callcenter']['denied_operator_view_roles']
        return cls(ticket, denied_operator_view_roles)

    def get_requests(self, request=None, limit=None, user=None, since=None,
                     until=None, staff_entry_binding=None, phone_number=None, **kwargs):
        return []

        if self._check_user_not_matched(user, phone_number):
            return []

        if staff_entry_binding is not None:
            return []

        devices = None
        if user:
            devices = (UserDevices.objects.using(cars.settings.DB_RO_ID)
                       .filter(user_id=user.id)
                       .only('device_id', 'user_id'))
            if not devices:
                return []

        filters = self._filter_params(devices, since, until, limit)
        if not filters:
            raise Exception('at least one filter must be provided')

        resp = requests.get(build_appmetrica_request(filters),
                            headers=self._get_headers(),
                            timeout=RESPONSE_TIMEOUT_SECONDS)
        resp.raise_for_status()
        return self._process_response(user, resp.json())

    def _filter_params(self, devices, since, until, limit):
        res = {}

        if devices:
            filters = ['device=={}'.format(sip(d.device_id))
                       for d in devices]
            res['filters'] = ' OR '.join(filters)

        since = dt.datetime.now() - DEFAULT_VIEW_PERIOD
        res['date1'] = datetime_to_date_str(since)

        if until is not None:
            res['date2'] = datetime_to_date_str(until)

        if limit:
            res['limit'] = limit

        return res

    def _process_response(self, user, json_data):
        formatted_user = (self.user_processing_helper.format_user(user)
                          if user else None)

        entries = []
        request_description = {
            'id': None,
            'origin': None,
            'queue': self.PUSH_QUEUE_NAME,
            'type': self.REQUEST_TYPE.value,
        }
        sender = StaffUserEntrySerializer(None).data
        sender['print_name'] = 'appmetrica'
        for d in json_data['data']:
            campaign_name = d['dimensions'][0]['name']
            data = {'message': campaign_name}
            for index, m in enumerate(d['metrics'][0]):
                for k in range(int(float(m))):
                    recived_count = int(float(d['metrics'][1][index]))
                    status = ('serviced' if k <= recived_count
                              else 'not_serviced')
                    day_str = json_data['time_intervals'][index][0]
                    send_time = dt.datetime.strptime(day_str, '%Y-%m-%d')
                    # Set middle day time because we have
                    # only date in response.
                    send_time = send_time.replace(hour=12, minute=0)
                    time_enter = time_exit = send_time.timestamp()

                    entry = AggregatedResponseEntry(
                        user=formatted_user,
                        time_enter=time_enter,
                        time_exit=time_exit,
                        time_connect=None,
                        connect_trial_count=None,
                        duration=None,
                        duration_print='',
                        status=status,
                        operators=[sender],
                        request=request_description,
                        message=campaign_name,
                        data_url=None,
                        data=data,
                        phone=None,
                        tags=[],
                    )
                    entries.append(entry._asdict())
        return entries
