import collections
import datetime
import json
import logging

from cars.core.telephony import TelephonyHelperBase
from cars.core.util import datetime_helper, phone_number_helper
from cars.callcenter.core import StaffInfoHelper
from cars.users.models import User

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.models.audiotele_stats import AudioteleCallDirection

from .syncing_helper import AudioteleDailyStatSyncHelper


LOGGER = logging.getLogger(__name__)


class AudioteleAPIHelper(TelephonyHelperBase):
    staff_info_helper = StaffInfoHelper.make_default()

    CallEnterEntry = collections.namedtuple(
        'CallEnterEntry', ('direction', 'time_enter', 'time_connect',
                           'phone', 'agent', 'staff_entry_binding', 'call_id', 'meta_info')
    )

    CallExitEntry = collections.namedtuple(
        'CallExitEntry', ('direction', 'time_enter', 'time_connect', 'time_exit', 'duration', 'is_answered',
                          'phone', 'agent', 'staff_entry_binding', 'call_id', 'meta_info')
    )

    def make_call_enter_entry(self, **raw_entry):
        meta_info = {}

        time_enter = datetime_helper.timestamp_ms_to_datetime(raw_entry['time_enter'])

        original_phone = raw_entry['phone']
        phone = phone_number_helper.normalize_phone_number(original_phone)

        if phone is None:
            meta_info['original_phone'] = original_phone

        raw_agent = raw_entry['agent']

        agent = raw_agent.lower() if raw_agent is not None and raw_agent != 'null' else None
        staff_entry_binding = self.staff_info_helper.get_agent_entry(username=agent)

        call_id = raw_entry['call_id']

        entry = self.CallEnterEntry(
            direction=AudioteleCallDirection.INCOMING,  # all entries are incoming now
            time_enter=time_enter,
            time_connect=time_enter,  # only connected entries are reported now
            phone=phone,
            agent=agent,
            staff_entry_binding=staff_entry_binding,
            call_id=call_id,
            meta_info=(meta_info or None),
        )
        return entry

    def make_call_exit_entry(self, **raw_entry):
        meta_info = {}

        direction = AudioteleCallDirection(raw_entry['direction'])

        time_field_names = ('time_enter', 'time_connect', 'time_exit')
        time_enter, time_connect, time_exit = [
            datetime_helper.timestamp_to_datetime(raw_entry[x], is_local=True) if raw_entry[x] != 'null' else None
            for x in time_field_names
        ]

        if time_exit is None:
            LOGGER.error(
                'entry with id {} has incorrect call finish time: {}'
                .format(raw_entry.get('call_id', None), json.dumps(raw_entry))
            )
            return None

        is_answered = (raw_entry['is_answered'].lower() == 'true')  # as string

        duration = int(raw_entry['duration']) if is_answered else None

        original_phone = raw_entry['phone']
        phone = phone_number_helper.normalize_phone_number(original_phone)

        if phone is None:
            meta_info['original_phone'] = original_phone

        # process daily stat and webhook data in a different way
        raw_agent = raw_entry['agent_login'] if 'agent_login' in raw_entry else raw_entry['agent']

        agent = raw_agent.lower() if (raw_agent is not None and raw_agent != 'null') else None

        if agent is not None:
            staff_entry_binding = self.staff_info_helper.get_agent_entry(username=agent)
        else:
            meta_info['original_agent_id'] = raw_entry.get('agent', None)
            meta_info['original_agent_print_name'] = raw_entry.get('agent_print_name', None)
            staff_entry_binding = None

        call_id = raw_entry['call_id']

        entry = self.CallExitEntry(
            direction=direction, time_enter=time_enter, time_connect=time_connect, time_exit=time_exit,
            duration=duration, is_answered=is_answered,
            phone=phone, agent=agent, staff_entry_binding=staff_entry_binding, call_id=call_id,
            meta_info=(meta_info or None),
        )
        return entry

    def get_generalized_user_data(self, phone, staff_entry):
        from_number = phone_number_helper.normalize_phone_number(phone)

        if from_number:
            # do not add bad formatted entries
            from_user = (
                User.objects.filter(phone=from_number)
                    .order_by('status')  # make "active" first if exists
                    .first()
            )
        else:
            from_user = None

        _to_user_binding = staff_entry

        if _to_user_binding is not None:
            to_user = _to_user_binding.user
            to_number = _to_user_binding.work_phone
        else:
            to_user = None
            to_number = None

        return from_number, to_number, from_user, to_user


class AudioteleDailyStatAPIHelper(AudioteleAPIHelper):
    API_URL = 'https://cc-apps.audiotele.ru/yandex_drive-1.0.0p/api/direct/{}'
    API_DATE_FORMAT = '%Y-%m-%d'

    DEFAULT_LAST_PROCESSED_DATE = datetime.date(2018, 10, 14)

    staff_info_helper = StaffInfoHelper.make_default()

    def __init__(self, username, password, request_timeout=15):
        super().__init__(request_timeout=request_timeout)
        self._credentials = (username, password)
        self._session.verify = False

        self._time_sync_helper = AudioteleDailyStatSyncHelper.from_settings()

    @classmethod
    def from_settings(cls):
        username = settings['audiotele']['request_username']
        password = settings['audiotele']['request_password']
        request_timeout = settings['audiotele']['request_timeout']
        return cls(username=username, password=password, request_timeout=request_timeout)

    def iter_data(self):
        for date_to_process in self._time_sync_helper.iter_dates_to_process():
            yield self.iter_day_data(date_to_process)

    def iter_day_data(self, date_to_process):
        api_url = self._make_api_url(date_to_process)
        data = self._perform_request(api_url, auth=self._credentials, raise_for_status=True, verify=False)

        for x in data['DATA']:
            raw_entry = self.make_call_exit_entry(**x)
            if raw_entry is not None:
                yield raw_entry

    def _make_api_url(self, date_to_process):
        day_str = date_to_process.strftime(self.API_DATE_FORMAT)
        api_url = self.API_URL.format(day_str)
        return api_url
