import datetime
import logging

from cars.request_aggregator.models.audiotele_stats import AudioteleIncomingCallEntry, AudioteleCallAction

from .api_helper import AudioteleDailyStatAPIHelper
from .data_moving_helper import AudioteleCallTrackManager
from .mixins import AudioteleReporterMixin
from .syncing_helper import AudioteleCallTrackCollectingSyncHelper


LOGGER = logging.getLogger(__name__)


class AudioteleCallStatCollectingHelper(AudioteleReporterMixin):
    api_helper = AudioteleDailyStatAPIHelper.from_settings()

    @classmethod
    def from_settings(cls):
        return cls()

    def update_data(self):
        total_count = 0

        for date_data in self.api_helper.iter_data():
            total_count += self._process_entries(list(date_data))

        return total_count

    def _process_entries(self, entries):
        call_id_collection = (e.call_id for e in entries)
        existing_call_id_collection = set(
            AudioteleIncomingCallEntry.objects
            .filter(
                related_call_id__in=call_id_collection,
                action=AudioteleCallAction.FINISH.value
            )
            .values_list('related_call_id', flat=True)
        )

        absent_entries = (e for e in entries if e.call_id not in existing_call_id_collection)

        entries_to_create = [
            AudioteleIncomingCallEntry(
                direction=e.direction.value,
                action=AudioteleCallAction.FINISH.value,
                time_enter=e.time_enter,
                time_connect=e.time_connect,
                time_exit=e.time_exit,
                related_call_id=e.call_id,
                is_answered=e.is_answered,
                duration=e.duration,
                phone=e.phone,
                agent=e.agent,
                staff_entry_binding=e.staff_entry_binding,
                meta_info=e.meta_info,
            )
            for e in absent_entries
        ]

        AudioteleIncomingCallEntry.objects.bulk_create(entries_to_create)

        return len(entries_to_create)


class AudioteleCallTrackCollectingHelper(object):
    FOLDER_PATH_TEMPLATE = '/files/Voice/%Y%m%d'

    def __init__(self):
        self._time_sync_helper = AudioteleCallTrackCollectingSyncHelper.from_settings()

    @classmethod
    def from_settings(cls):
        return cls()

    def update_data(self):
        for date in self._time_sync_helper.iter_dates_to_process():
            path = self._get_folder_path(date)
            call_track_manager = AudioteleCallTrackManager(path)
            call_track_manager.process(re_raise=True)

    def _get_folder_path(self, date):
        return datetime.date.strftime(date, self.FOLDER_PATH_TEMPLATE)
