import logging

from cars.core.mds.client import MDSClient, EncryptedMDSClient
from cars.settings import CALLCENTER as callcenter_settings, REQUEST_AGGREGATOR as request_aggregator_settings

from cars.request_aggregator.core.ftp_helper import SFTPFileDownloader, SFTPDownloadFileManager
from cars.request_aggregator.models import AudioteleCallTrackEntry


LOGGER = logging.getLogger(__name__)


class AudioteleMDSHelper(object):
    def __init__(self, client, bucket_name):
        self._client = client
        self._bucket_name = bucket_name

    @classmethod
    def from_settings(cls):
        mds_authorization_extra_settings = callcenter_settings['mds']
        mds_client = MDSClient.from_settings(**mds_authorization_extra_settings)

        mds_connection_settings = request_aggregator_settings['audiotele']['mds_connection']

        if mds_connection_settings['encrypt']:
            aes_key = mds_connection_settings['aes_key']
            mds_client = EncryptedMDSClient(mds_client, aes_key)

        bucket_name = mds_connection_settings['bucket_name']
        return cls(mds_client, bucket_name)

    def put_object(self, track_entry, file_content):
        assert isinstance(track_entry, AudioteleCallTrackEntry)
        mds_file_key = track_entry.mds_key
        response = self._client.put_object(bucket=self._bucket_name, key=mds_file_key, body=file_content)
        return response

    def get_object(self, track_entry):
        assert isinstance(track_entry, AudioteleCallTrackEntry)
        response = self._client.get_object(
            bucket=self._bucket_name, key=track_entry.mds_key
        )
        return response

    def get_object_content(self, track_entry):
        assert isinstance(track_entry, AudioteleCallTrackEntry)
        content = self._client.get_object_content(
            bucket=self._bucket_name, key=track_entry.mds_key
        )
        return content


class AudioteleFileDownloader(SFTPFileDownloader):
    @classmethod
    def from_settings(cls, root_dir):
        audiotele_ftp_connection_settings = request_aggregator_settings['audiotele']['ftp_connection']
        return cls(root_dir=root_dir, **audiotele_ftp_connection_settings)


class AudioteleCallTrackManager(SFTPDownloadFileManager):
    def __init__(self, root_dir):
        super().__init__(root_dir, AudioteleFileDownloader, AudioteleMDSHelper)

    def _filter_entry(self, track_entry):
        return not AudioteleCallTrackEntry.objects.filter(file_name=track_entry.file_name).exists()

    def _make_track_entry(self, file_description):
        track_entry = AudioteleCallTrackEntry(
            file_name=file_description.filename,
            file_attrs={
                'size': file_description.st_size,
                'mtime': file_description.st_mtime,
            },
        )
        return track_entry
