import datetime
import json
import logging

import cars.settings

from cars.core.util import datetime_helper
from cars.request_aggregator.core.data_exporting_helper import DataExportingSyncHelper, DataExportingHelper
from cars.request_aggregator.core.request_time_sync_helper import RequestStateSyncHelper
from cars.request_aggregator.models.audiotele_stats import AudioteleIncomingCallEntry, AudioteleCallTrackEntry
from cars.request_aggregator.models.call_center_common import CallStatSyncStatus, SyncOrigin

LOGGER = logging.getLogger(__name__)


class AudioteleCCDataExportingSyncHelper(DataExportingSyncHelper):
    entry_model_cls = AudioteleIncomingCallEntry


class AudioteleCCDataExportingHelper(DataExportingHelper):
    _sync_helper_cls = AudioteleCCDataExportingSyncHelper

    def _make_export_entry(self, entry):
        data = {
            'agent': entry.agent,
            'callId': entry.related_call_id,

            'id': str(entry.id),
            'phone': self.get_original_phone(entry),
            'agentUsername': self.get_agent_username(entry),
            'metaInfo': json.dumps(entry.meta_info) if entry.meta_info else None,

            'timeEnter': self.time_to_repr(entry.time_enter),
            'timeConnect': self.time_to_repr(entry.time_connect),
            'timeExit': self.time_to_repr(entry.time_exit),
            'duration': entry.duration,

            'action': entry.action,
            'direction': entry.direction,
        }

        return data


class AudioteleCCTrackDataExportingSyncHelper(RequestStateSyncHelper):
    entry_model_cls = AudioteleCallTrackEntry

    MAX_ENTRIES_COUNT = 5000

    @classmethod
    def from_settings(cls, sync_status_origin):
        return cls(
            stat_sync_origin=sync_status_origin,
            default_state={'entry_filename_lo': None, 'entry_filename_hi': None}
        )

    def _get_state(self, sync_entry):
        _, entry_filename_lo = sync_entry.data['entry_filename_lo'], sync_entry.data['entry_filename_hi']

        if not self._is_matching_completed():
            return entry_filename_lo, entry_filename_lo

        qs = self._get_qs()
        if entry_filename_lo is not None:
            qs = qs.filter(file_name__gt=entry_filename_lo)

        qs = qs.order_by('file_name').values_list('file_name', flat=True)[:self.MAX_ENTRIES_COUNT]
        entry_filename_hi = max(qs, default=entry_filename_lo)

        return entry_filename_lo, entry_filename_hi

    def _get_qs(self):
        return self.entry_model_cls.objects.using(cars.settings.DB_RO_ID).all()

    def _is_matching_completed(self):
        last_track_matching = next(iter(
            CallStatSyncStatus.objects.filter(origin=SyncOrigin.CC_AUDIOTELE_TRACK_BINDINGS.value)
            .values_list('last_data_sync_time', flat=True)
        ), None)

        if last_track_matching is not None:
            return last_track_matching.date() + datetime.timedelta(days=1) == datetime_helper.utc_now().date()
        return False

    def _update_state(self, sync_entry, state=None):
        entry_filename_lo, entry_filename_hi = state

        sync_entry.data['entry_filename_lo'] = entry_filename_lo
        sync_entry.data['entry_filename_hi'] = entry_filename_hi

        # try to get more entries
        has_more = (entry_filename_lo != entry_filename_hi)
        return has_more


class AudioteleCCTrackDataExportingHelper(DataExportingHelper):
    _sync_helper_cls = AudioteleCCTrackDataExportingSyncHelper

    BUCKET_NAME = 'carsharing-support-audiotele'
    TRACK_URL_TEMPLATE = (
        'https://carsharing.yandex-team.ru/api/request_aggregator/v1/audiotele/tracks/?track_id={track_id}'
    )

    def _iter_data_to_store(self, sync_state):
        entry_filename_lo, entry_filename_hi = sync_state

        if entry_filename_lo != entry_filename_hi:
            entries = self._iter_entries(entry_filename_lo, entry_filename_hi)
            yield from (self._make_export_entry(entry) for entry in entries)

    def _iter_entries(self, entry_filename_lo, entry_filename_hi):
        entries = self._entry_model_cls.objects.using(cars.settings.DB_RO_ID).all()
        if entry_filename_lo is not None:
            entries = entries.filter(file_name__gt=entry_filename_lo)
        entries = entries.filter(file_name__lte=entry_filename_hi)
        yield from entries

    def _make_export_entry(self, entry):
        def try_convert(value, converter=str):
            return converter(value) if value is not None else None

        data = {
            'id': str(entry.id),
            'callId': try_convert(entry.call_entry_id),
            'fileName': entry.file_name,
            'fileAttributes': try_convert(entry.file_attrs, json.dumps),
            'externalCallId': entry.call_id,
            'timestamp': self.time_to_repr(entry.time_id),
            'rawPhone': entry.raw_phone,
            'phone': entry.phone,
            'mdsKey': entry.mds_key,
            'mdsBucket': self.BUCKET_NAME,
            'url': self.TRACK_URL_TEMPLATE.format(track_id=entry.id),
        }
        return data
