import enum
import itertools
import logging

from django.db.models import Q
from django.urls import reverse

import cars.settings

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.core.util import datetime_helper
from cars.callcenter.core import StaffInfoHelper
from cars.settings import REQUEST_AGGREGATOR as request_aggregator_settings

from cars.request_aggregator.core.common_helper import collection_to_mapping
from cars.request_aggregator.core.phone_binding_helper import UserProcessingHelper
from cars.request_aggregator.core.request_tags import TagDescriptionHelper
from cars.request_aggregator.models.audiotele_stats import (
    AudioteleIncomingCallEntry, AudioteleCallTrackEntry, AudioteleCallDirection, AudioteleCallAction
)
from cars.request_aggregator.models.call_tags import RequestOriginType
from cars.request_aggregator.serializers.common import AggregatedResponseEntry, RequestType

from .data_moving_helper import AudioteleMDSHelper

LOGGER = logging.getLogger(__name__)


class AudioteleCallStatus(enum.Enum):
    PENDING = 'pending'
    CONNECTED = 'connected'
    SERVICED = 'serviced'
    NOT_SERVICED = 'not_serviced'


class AudioteleProcessingHelper(object):
    REQUEST_TYPE = RequestType.phone

    INCOMING_QUEUE_NAME = 'atele_incoming'
    OUTGOING_QUEUE_NAME = 'atele_outgoing'

    # api_url starts with slash so it's not required
    DATA_URL_TEMPLATE = 'https://carsharing.yandex-team.ru{api_url}?track_id={track_id}'

    staff_info_helper = StaffInfoHelper.make_default()
    tag_description_helper = TagDescriptionHelper()
    saas_client = SaasDriveAdminClient.from_settings()
    mds_helper = AudioteleMDSHelper.from_settings()
    user_processing_helper = UserProcessingHelper.from_settings()

    def __init__(self, call_track_view_actions, denied_operator_view_roles, denied_data_view_roles):
        self._call_track_view_actions = call_track_view_actions
        self._denied_operator_view_roles = denied_operator_view_roles
        self._denied_data_view_roles = denied_data_view_roles

    @classmethod
    def from_settings(cls):
        call_track_view_actions = request_aggregator_settings['callcenter']['call_track_view_actions']
        denied_operator_view_roles = request_aggregator_settings['callcenter']['denied_operator_view_roles']
        denied_data_view_roles = request_aggregator_settings['callcenter']['denied_data_view_roles']
        return cls(call_track_view_actions, denied_operator_view_roles, denied_data_view_roles)

    def check_request_type(self, request_type_collection=None):
        return not request_type_collection or self.REQUEST_TYPE.value in request_type_collection

    def get_call_url(self, *, track=None, track_id=None):
        call_url = None

        if track_id is None and track is not None:
            track_id = track.id

        if track_id is not None:
            api_url = reverse('request_aggregator:audiotele-tracks')
            call_url = self.DATA_URL_TEMPLATE.format(api_url=api_url, track_id=track_id)

        return call_url

    def get_requests(self, request=None, limit=None, **kwargs):
        if self._check_phone_not_matched(**kwargs):
            return []

        filters = self._make_call_filters(**kwargs)

        if not filters and limit is None:
            raise Exception('at least one filter must be provided')

        limit = 2 * limit if limit is not None else None  # to get both call start and finish entries

        call_entries = (
            AudioteleIncomingCallEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(**filters)
            .order_by('-time_enter', '-id')
            [:limit]
        )

        grouped_call_entries = collection_to_mapping(call_entries, attr_key='related_call_id')

        entry_tags_mapping = self.tag_description_helper.get_call_tags(
            (e.id for e in call_entries),
            (
                Q(request_origin=RequestOriginType.AUDIOTELE_INCOMING.value) |
                Q(request_origin=RequestOriginType.AUDIOTELE_OUTGOING.value)
            )
        )

        users_mapping = self._make_call_users_mapping(call_entries, request, kwargs)

        access_mapping = self._make_access_mapping(request)

        response_entries_as_mappings = self._transform_raw_entries(
            grouped_call_entries.values(), entry_tags_mapping, users_mapping, access_mapping, limit
        )

        return response_entries_as_mappings

    def _check_phone_not_matched(self, user=None, phone_number=None, **kwargs):
        # user phone number is not specified neither explicitly nor present in user info
        return user is not None and phone_number is None

    def _make_call_filters(self, phone_number=None, since=None, until=None, staff_entry_binding=None, **kwargs):
        filters = {}

        if phone_number is not None:
            filters['phone'] = phone_number

        if since is not None:
            filters['time_enter__gte'] = since

        if until is not None:
            filters['time_enter__lt'] = until

        if staff_entry_binding is not None:
            filters['staff_entry_binding'] = staff_entry_binding

        return filters

    def _make_call_users_mapping(self, call_entries, request, kwargs):
        if 'user' in kwargs:
            raw_entry_phone_bindings = {None: kwargs['user']}
            self.user_processing_helper.filter_deleted_users(raw_entry_phone_bindings, request)

            entry_phone_bindings = self.user_processing_helper.format_users(
                default=raw_entry_phone_bindings[None], update_from_key='phone'
            )

        else:
            raw_entry_phone_bindings = self.user_processing_helper.get_entries_phone_bindings(call_entries)
            self.user_processing_helper.filter_deleted_users(raw_entry_phone_bindings, request)

            entry_phone_bindings = self.user_processing_helper.format_users(
                raw_entry_phone_bindings, update_from_key='phone'
            )

        users_mapping = {
            call_entry.id: entry_phone_bindings[call_entry.phone]
            for call_entry in call_entries
        }

        return users_mapping

    def _transform_raw_entries(self, grouped_call_entries, entry_tags_mapping, users_mapping, access_mapping, limit):
        entry_limit = range(limit) if limit is not None else itertools.count()

        response_entries_as_mappings = [
            self._process_call_entry(entries_group, entry_tags_mapping, users_mapping, access_mapping)._asdict()
            for idx, entries_group in zip(entry_limit, grouped_call_entries)
        ]
        return response_entries_as_mappings

    def _process_call_entry(self, entries_group, entry_tags_mapping, users_mapping, access_mapping):
        call_start_data = call_finish_data = None

        for entry in entries_group:  # NB. Some entries could be duplicated
            if entry.action == AudioteleCallAction.START.value:
                call_start_data = self._get_call_start_specific_data(
                    entry, entry_tags_mapping, users_mapping, access_mapping
                )
            elif (
                    entry.action == AudioteleCallAction.FINISH.value
                    or entry.action is None  # remove when action will be None
            ):
                has_data_track_url = call_finish_data and call_finish_data['data']['track_url'] is not None
                if not has_data_track_url:  # select one with call track bound
                    call_finish_data = self._get_call_finish_specific_data(
                        entry, entry_tags_mapping, users_mapping, access_mapping
                    )
            else:
                raise RuntimeError('unexpected call action')

        if call_start_data is not None and call_finish_data is not None:
            call_finish_data['tags'].extend(call_start_data['tags'])

            # finish entry may lack of staff entry if call is shorter than 2 seconds
            if call_finish_data['_staff_entry'] is None and call_start_data['_staff_entry'] is not None:
                call_finish_data['operators'] = call_start_data['operators']

            call_data = call_finish_data
        else:
            call_data = call_start_data or call_finish_data

        call_data.pop('_staff_entry')

        return AggregatedResponseEntry(**call_data)

    def _get_call_start_specific_data(self, entry, entry_tags_mapping, users_mapping, access_mapping):
        # currently all calls are connected
        if entry.time_connect is not None:
            time_connect = entry.time_connect.timestamp()
            status = AudioteleCallStatus.CONNECTED
        else:
            time_connect = None
            status = AudioteleCallStatus.PENDING

        call_data = {
            'time_enter': entry.time_enter.timestamp(),
            'time_connect': time_connect,
            'time_exit': None,
            'status': status.value,
            'data': {'track_url': None},  # entries are bind after call is finished
            'message': None,  # deprecated
            'data_url': None,  # deprecated
        }

        generic_data = self._get_call_generic_data(entry, entry_tags_mapping, users_mapping, access_mapping)
        call_data.update(generic_data)

        return call_data

    def _get_call_finish_specific_data(self, entry, entry_tags_mapping, users_mapping, access_mapping):
        if entry.time_connect is not None:
            time_connect = entry.time_connect.timestamp()
            status = AudioteleCallStatus.SERVICED
        else:
            time_connect = None
            status = AudioteleCallStatus.NOT_SERVICED

        if access_mapping['has_access_to_call_id']:
            data_url = self.get_call_url(track=getattr(entry, 'track', None))
        else:
            data_url = None

        call_data = {
            'time_enter': entry.time_enter.timestamp(),
            'time_connect': time_connect,
            'time_exit': entry.time_exit.timestamp(),
            'status': status.value,
            'data': {'track_url': data_url},
            'message': None,  # deprecated
            'data_url': data_url,  # deprecated
        }

        generic_data = self._get_call_generic_data(entry, entry_tags_mapping, users_mapping, access_mapping)
        call_data.update(generic_data)

        return call_data

    def _get_call_generic_data(self, entry, entry_tags_mapping, users_mapping, access_mapping):
        duration = entry.duration
        duration_print = datetime_helper.duration_to_str(duration)

        staff_entry = self.staff_info_helper.get_agent_entry(username=entry.agent)

        agent = self.staff_info_helper.format_staff_entry(staff_entry)

        if access_mapping['has_access_to_operators']:
            operators = [agent]
        else:
            operators = []

        phone = str(entry.phone) if entry.phone is not None else entry.phone

        tags = entry_tags_mapping.get(str(entry.id), [])

        if entry.direction == AudioteleCallDirection.OUTGOING.value:
            queue = self.OUTGOING_QUEUE_NAME
            request_origin = RequestOriginType.AUDIOTELE_OUTGOING.name
        else:
            queue = self.INCOMING_QUEUE_NAME
            request_origin = RequestOriginType.AUDIOTELE_INCOMING.name

        request_description = {
            'id': entry.id,
            'origin': request_origin,
            'queue': queue,
            'type': self.REQUEST_TYPE.value,
        }

        user = users_mapping.get(entry.id, None)

        call_data = {
            'connect_trial_count': 1,
            'duration': duration,
            'duration_print': duration_print,
            'phone': phone,
            'operators': operators,
            'tags': tags,
            'request': request_description,
            'user': user,
            '_staff_entry': staff_entry,  # extra info, to be removed
        }
        return call_data

    def _make_access_mapping(self, request=None):
        access_mapping = {}

        if request is not None:
            has_access_to_call_id = (
                self.saas_client.check_user_action(
                    *self._call_track_view_actions, request=request, require_all=False
                ) and
                not self.saas_client.check_user_role(
                    *self._denied_data_view_roles, request=request, require_all=False
                )
            )
            has_access_to_operators = not self.saas_client.check_user_role(
                *self._denied_operator_view_roles, request=request, require_all=False
            )
        else:
            has_access_to_call_id = False
            has_access_to_operators = False

        access_mapping['has_access_to_call_id'] = has_access_to_call_id
        access_mapping['has_access_to_operators'] = has_access_to_operators

        return access_mapping

    def get_call_track(self, *, track=None, track_id=None, external_call_id=None):
        assert (track is not None) or (track_id is not None) or (external_call_id is not None), 'either track or track_id or call id must be specified'

        if track is None:
            if track_id is not None:
                track = (
                    AudioteleCallTrackEntry.objects.using(cars.settings.DB_RO_ID)
                    .filter(id=track_id).first()
                )
            elif external_call_id is not None:
                call_ids = list(
                    AudioteleIncomingCallEntry.objects.using(cars.settings.DB_RO_ID)
                    .filter(related_call_id=external_call_id).values_list('id', flat=True)
                )
                if call_ids:
                    track = (
                        AudioteleCallTrackEntry.objects.using(cars.settings.DB_RO_ID)
                        .filter(call_entry_id__in=call_ids).first()
                    )
            else:
                assert False, 'invalid track id parameters'

        data = None
        if track is not None:
            data = self.mds_helper.get_object_content(track_entry=track)

        return data
