import logging
import operator
import time
from multiprocessing.dummy import Pool as ThreadPool

import requests
from django.db import connection, transaction
from django.db.models import Q

import cars.settings
from cars.core.util import datetime_helper

from cars.callcenter.core import StaffInfoHelper
from cars.request_aggregator.core.common_helper import collection_to_mapping
from cars.request_aggregator.models import AudioteleIncomingCallEntry, CallStatSyncStatus
from cars.request_aggregator.models.call_tags import RequestTagEntry, RequestOriginType


LOGGER = logging.getLogger(__name__)


class AudioteleServiceHelper(object):
    MAX_ENTRIES_LIMIT = 1000

    staff_info_helper = StaffInfoHelper.make_default()

    @classmethod
    def from_settings(cls):
        return cls()

    def update_staff_bindings(self, offset, count):
        limit = min(count, self.MAX_ENTRIES_LIMIT)

        with transaction.atomic(savepoint=False):
            entries = (
                AudioteleIncomingCallEntry.objects.select_for_update()
                .filter(
                    staff_entry_binding__isnull=True,
                    agent__isnull=False,
                )
                .order_by('id')
                [offset:offset + limit]
            )

            for entry in entries:
                agent_entry = self.staff_info_helper.get_agent_entry(username=entry.agent)
                if agent_entry is not None:
                    entry.staff_entry_binding = agent_entry
                    entry.save()

        total_numbers = len(entries)
        LOGGER.info('total numbers updated: {}'.format(total_numbers))

        return total_numbers

    def update_broken_entries(self):
        def format_broken_id(call_id):
            return '_'.join(('broken', call_id))

        broken_entry_ids = []
        dates_to_process = set()

        with transaction.atomic(savepoint=False):
            broken_entries = (
                AudioteleIncomingCallEntry.objects
                .select_for_update()
                .filter(
                    Q(phone__isnull=True) | Q(is_answered=True, agent__isnull=True),
                    ~Q(related_call_id__startswith='broken'),
                )
            )

            for e in broken_entries:
                broken_entry_ids.append(e.related_call_id)
                dates_to_process.add(e.time_enter.date())
                e.related_call_id = format_broken_id(e.related_call_id)
                e.save()

        sync_entry = CallStatSyncStatus.objects.get(origin=CallStatSyncStatus.Origin.CC_AUDIOTELE.value)

        for date in dates_to_process:
            since = until = datetime_helper.utc_localize(datetime.datetime.combine(date, datetime.time.min)).timestamp()
            sync_entry.append_extra_time_span(since, until)

        sync_entry.save()
        return

        # request the data again

        repaired_entry_ids = (
            format_broken_id(call_id)
            for call_id in (
                AudioteleIncomingCallEntry.objects
                .filter(related_call_id__in=broken_entry_ids)
                .values_list('related_call_id', flat=True)
            )
        )

        with transaction.atomic(savepoint=False):
            repaired_entries = AudioteleIncomingCallEntry.objects.filter(related_call_id__in=repaired_entry_ids)
            repaired_entries.delete()

    def remove_duplicates(self):
        def format_duplicated_id(call_id):
            return '_'.join(('duplicate', call_id))

        make_comparable_tuple = operator.attrgetter('time_enter', 'time_connect', 'time_exit', 'phone', 'agent')

        with connection.cursor() as cursor:
            cursor.execute(
                'SELECT DISTINCT related_call_id FROM audiotele_call_stats'
                ' GROUP BY related_call_id HAVING COUNT(related_call_id) > 1;'
            )
            duplicate_entry_ids = list(cursor.fetchall())

        duplicated_entries = collection_to_mapping(
            AudioteleIncomingCallEntry.objects.filter(related_call_id__in=duplicate_entry_ids),
            key=lambda x: (x.related_call_id, x.action)
        )

        entry_id_to_rename = []

        call_internal_ids = (e.id for g in duplicated_entries.values() for e in g)
        tagged_item_ids = set(
            RequestTagEntry.objects
            .filter(
                request_id__in=call_internal_ids,
                request_origin__in=(RequestOriginType.AUDIOTELE_OUTGOING.value, RequestOriginType.AUDIOTELE_INCOMING.value),
            )
            .values_list('request_id', flat=True)
        )

        for (call_id, action), entries in duplicated_entries.items():
            distinct_entry_count = len({make_comparable_tuple(x) for x in entries})

            if distinct_entry_count == 1:
                not_tagged_entries = [e.id for e in entries if e.id not in tagged_item_ids]

                tagged_entries_count = len(entries) - len(not_tagged_entries)

                if tagged_entries_count < 2:
                    entry_id_to_rename.extend(not_tagged_entries)
                else:
                    LOGGER.warning('multiple entries with same call id {} has been tagged'.format(call_id))

            else:
                LOGGER.warning('entries with related id {} marked as duplicated but they differ'.format(call_id))

        if entry_id_to_rename:
            LOGGER.info('renaming {} duplicates: {}'.format(len(entry_id_to_rename), entry_id_to_rename))

            with transaction.atomic(savepoint=False):
                for entry in AudioteleIncomingCallEntry.objects.filter(id__in=entry_id_to_rename):
                    entry.related_call_id = format_duplicated_id(entry.related_call_id)
                    entry.save()
