import datetime
import enum

from cars.core.util import datetime_helper
from cars.callcenter.models.call_assignment import CallAssignment, CallAssignmentCallCenter
from cars.request_aggregator.core.statistics import StatisticsHelperBase, register
from cars.request_aggregator.core.request_time_sync_helper import RequestTimeSyncHelper
from cars.request_aggregator.models.audiotele_stats import AudioteleIncomingCallEntry, AudioteleCallDirection
from cars.request_aggregator.models.call_center_common import SyncOrigin


@register('audiotele')
class AudioteleStatisticsHelper(StatisticsHelperBase):
    class Queues(enum.Enum):
        INCOMING = 'incoming'
        OUTGOING = 'outgoing'

    def __init__(self):
        super().__init__()
        self._sync_helper = RequestTimeSyncHelper(
            use_timestamp=False,
            stat_sync_origin=SyncOrigin.CC_AUDIOTELE_STAT_MONITORING,
            default_since=datetime_helper.utc_now(),
            request_lag=datetime.timedelta(minutes=1),
        )

    def collect(self):
        stats = {}
        for since, until in self._sync_helper.iter_time_span_to_process():
            self.update_collection(stats, self.get_call_entries_stat(since, until))
        return stats

    def get_call_entries_stat(self, since, until):
        queue_stats = {
            queue.value: self.get_default_queue_stat_collection()
            for queue in self.Queues
        }

        self._process_intro_call_stats(queue_stats, since, until)

        entries = self._get_call_entries(since, until)

        for entry in entries:
            stats, entry = self._prepare_call_entry(queue_stats, entry)
            self._update_entry_stat(entry, stats)

        self.annotate_call_duration(queue_stats)

        return queue_stats

    def _process_intro_call_stats(self, queue_stats, since, until):
        # outgoing data is provided post fact
        incoming_queue_stats = queue_stats[self.Queues.INCOMING.value]

        # none of calls can be pending - no info about that
        intro_entries_count = self._get_intro_entries_count(since, until)
        incoming_queue_stats['total'] += intro_entries_count

        on_air_entry_count = self._get_on_air_entries_count(since, until)
        incoming_queue_stats['serviced'] += on_air_entry_count

    def _get_intro_entries_count(self, since, until):
        intro_entries_count = (
            AudioteleIncomingCallEntry.objects
            .filter(time_exit__gte=since, time_exit__lt=until, direction=AudioteleCallDirection.INCOMING.value)
            .count()
        )
        return intro_entries_count

    def _get_on_air_entries_count(self, since, until):
        on_air_entry_ids = (
            CallAssignment.objects
            .filter(call_center=CallAssignmentCallCenter.AudioTele.value)
            .values_list('call_id', flat=True)
        )
        on_air_entry_count = len(set(on_air_entry_ids))
        return on_air_entry_count

    def _get_call_entries(self, since, until):
        entries = (
            AudioteleIncomingCallEntry.objects
            .filter(time_exit__gte=since, time_exit__lt=until)
            .values('related_call_id', 'time_enter', 'time_connect', 'time_exit', 'direction', 'duration')
        )
        return entries

    def _prepare_call_entry(self, queue_stats, entry):
        if entry['direction'] == AudioteleCallDirection.INCOMING.value:
            queue_name = 'incoming'
        elif entry['direction'] == AudioteleCallDirection.OUTGOING.value:
            queue_name = 'outgoing'
        else:
            raise RuntimeError('unknown queue')

        stats = queue_stats[queue_name]
        return stats, entry

    def _update_entry_stat(self, entry, stats):
        stats['total_managed'] += 1

        if entry['time_connect'] is not None:
            stats['completed'] += 1
        else:
            stats['not_serviced'] += 1

        if entry['duration'] is not None:
            stats['durations'].append(entry['duration'])

        return stats
