import logging
import operator

from django.db import transaction

from cars.request_aggregator.core.common_helper import collection_to_mapping

from cars.request_aggregator.models.audiotele_stats import (
    AudioteleIncomingCallEntry, AudioteleCallTrackEntry, AudioteleCallAction
)

from .syncing_helper import AudioteleCallTrackBindingSyncHelper


LOGGER = logging.getLogger(__name__)


class AudioteleCallTrackBindingHelper(object):
    MAX_ENTRIES_BATCH = 500
    DAYS_LAG = 1  # some entries are received during the next day

    def __init__(self):
        self._time_sync_helper = AudioteleCallTrackBindingSyncHelper.from_settings()

    @classmethod
    def from_settings(cls):
        return cls()

    def update_track_bindings(self):
        for date in self._time_sync_helper.iter_dates_to_process(days_lag=self.DAYS_LAG):
            total_performed = offset = 0
            tracks_to_bind_count = self._get_tracks_to_bind_queryset(date).count()

            for _ in range(0, tracks_to_bind_count, self.MAX_ENTRIES_BATCH):
                count = min(self.MAX_ENTRIES_BATCH, tracks_to_bind_count - total_performed)
                total_performed, total_updated = self._update_track_bindings(date, offset, count)
                offset += (total_performed - total_updated)

    def _get_tracks_to_bind_queryset(self, date):
        date_filter = date.strftime('_%Y-%m-%d_')
        tracks_to_bind = (
            AudioteleCallTrackEntry.objects
            .filter(call_entry__isnull=True, file_name__contains=date_filter)
        )
        return tracks_to_bind

    @transaction.atomic(savepoint=False)
    def _update_track_bindings(self, date, offset=0, count=MAX_ENTRIES_BATCH):
        total_performed = -1
        total_updated = 0

        tracks_to_bind = self._get_tracks_to_bind_queryset(date)

        tracks_to_update = (
            tracks_to_bind
            .select_for_update()
            .order_by('id')
            [offset:offset + count]
        )

        related_call_entries = collection_to_mapping(
            AudioteleIncomingCallEntry.objects.filter(
                related_call_id__in=(t.call_id for t in tracks_to_update),
                action=AudioteleCallAction.FINISH.value,
            ),
            attr_key='related_call_id'
        )

        for total_performed, t in enumerate(tracks_to_update):
            call_entries = related_call_entries.get(t.call_id, [])

            if not call_entries:
                LOGGER.warning('no entries found for call id {}'.format(t.call_id))
                continue

            if len(call_entries) > 1:
                LOGGER.warning(
                    'expected exactly one entry for call id {} but found {} - binding the latest one'
                    .format(t.call_id, len(call_entries))
                )
                call_entries.sort(key=operator.attrgetter('id'), reverse=True)

            call_entry = call_entries[0]

            if not hasattr(call_entry, 'track'):
                if str(t.phone) != str(call_entry.phone):
                    LOGGER.warning(
                        'track\'s {} phone number ({}) does not match call entry\'s {} one ({})'
                        .format(t.id, t.phone, call_entry.id, call_entry.phone)
                    )
                else:
                    t.call_entry = call_entry
                    t.save()
                    total_updated += 1
            else:
                LOGGER.warning(
                    'duplicate found: track id - {}, track file name - {}, '
                    'found call entry id - {}, found track id - {}'
                    .format(t.id, t.file_name, call_entry.id, call_entry.track.id)
                )

        return (total_performed + 1), total_updated
