import json
import logging

from cars.callcenter.core import StaffInfoHelper
from cars.callcenter.models.call_assignment import CallAssignment, CallAssignmentCallCenter

from cars.request_aggregator.core.common_helper import atomic_with_retries
from cars.request_aggregator.models.audiotele_stats import (
    AudioteleIncomingCallEntry, AudioteleCallDirection, AudioteleCallAction
)

from .api_helper import AudioteleAPIHelper
from .mixins import AudioteleReporterMixin


LOGGER = logging.getLogger(__name__)


class AudioteleCallAssignmentHelper(AudioteleReporterMixin):
    api_helper = AudioteleAPIHelper()

    CC = CallAssignmentCallCenter.AudioTele.value
    QUEUE = 'audiotele-incoming-default'

    def register_call_entries(self, processed_entries):
        entries_to_register = []

        for e in processed_entries:
            time_enter = e.time_enter
            time_connect = e.time_connect
            call_id = e.call_id

            from_number, to_number, from_user, to_user = self.api_helper.get_generalized_user_data(
                e.phone, e.staff_entry_binding
            )

            assignment_entry = CallAssignment(
                from_number=from_number,
                to_number=to_number,
                from_user=from_user,
                to_user=to_user,
                added_at=time_enter,
                connected_at=time_connect,
                call_id=call_id,
                call_center=self.CC,
                queue=self.QUEUE,
            )

            entries_to_register.append(assignment_entry)

            LOGGER.info('call start ({}): call_id - {}, time_enter - {}, phone - {}, agent - {}'
                        .format(self.CC, e.call_id, e.time_enter, e.phone, e.agent))

        self._create_entries_atomic(entries_to_register)
        self._upsert_call_assignment_entries_in_tags(entries_to_register, extra_mark='webhook')

    @atomic_with_retries
    def _create_entries_atomic(self, entries_to_register):
        CallAssignment.objects.bulk_create(entries_to_register)

    def remove_call_entries(self, processed_entries):
        entries_to_remove = []

        for e in processed_entries:
            if e.direction == AudioteleCallDirection.INCOMING:
                call_id = e.call_id

                assignment_entries = CallAssignment.objects.filter(
                    call_center=CallAssignmentCallCenter.AudioTele.value,
                    call_id=call_id,
                )
                entries_to_remove.extend(assignment_entries)
            elif e.direction == AudioteleCallDirection.OUTGOING:
                pass  # outgoing entries start is not reported now
            else:
                raise Exception('invalid audiotele call direction: {}'.format(e.direction))

            LOGGER.info('call finish ({}): call_id - {}, time_enter - {}, phone - {}, agent - {}, direction - {}'
                        .format(self.CC, e.call_id, e.time_enter, e.phone, e.agent, e.direction))

        self._remove_call_assignment_entries_in_tags(entries_to_remove, extra_mark='webhook')
        self._remove_entries_atomic(entries_to_remove)

    @atomic_with_retries
    def _remove_entries_atomic(self, entries_to_remove):
        for query in entries_to_remove:
            query.delete()


class AudioteleCallRegisteringHelper(AudioteleReporterMixin):
    staff_info_helper = StaffInfoHelper.make_default()
    api_helper = AudioteleAPIHelper()

    def process_call_start(self, processed_entries):
        existing_call_ids = {
            AudioteleIncomingCallEntry.objects
            .filter(
                related_call_id__in=(e.call_id for e in processed_entries),
                action=AudioteleCallAction.START.value,
            )
            .values_list('related_call_id', flat=True)
        }

        entries = [
            AudioteleIncomingCallEntry(
                direction=e.direction.value,
                action=AudioteleCallAction.START.value,
                time_enter=e.time_enter,
                time_connect=e.time_connect,
                time_exit=None,
                related_call_id=e.call_id,
                is_answered=False,
                duration=None,
                phone=e.phone,
                agent=e.agent,
                staff_entry_binding=e.staff_entry_binding,
                meta_info=e.meta_info,
            )
            for e in processed_entries
            if e.call_id not in existing_call_ids
        ]
        self._create_entries_atomic(entries)

    def process_call_end(self, processed_entries):
        existing_call_ids = {
            AudioteleIncomingCallEntry.objects
            .filter(
                related_call_id__in=(e.call_id for e in processed_entries),
                action=AudioteleCallAction.FINISH.value,
            )
            .values_list('related_call_id', flat=True)
        }

        entries = [
            AudioteleIncomingCallEntry(
                direction=e.direction.value,
                action=AudioteleCallAction.FINISH.value,
                time_enter=e.time_enter,
                time_connect=e.time_connect,
                time_exit=e.time_exit,
                related_call_id=e.call_id,
                is_answered=e.is_answered,
                duration=e.duration,
                phone=e.phone,
                agent=e.agent,
                staff_entry_binding=e.staff_entry_binding,
                meta_info=e.meta_info,
            )
            for e in processed_entries
            if e.call_id not in existing_call_ids
        ]
        self._create_entries_atomic(entries)

    @atomic_with_retries
    def _create_entries_atomic(self, entries):
        AudioteleIncomingCallEntry.objects.bulk_create(entries)


class AudioteleWebHookHelper(object):
    api_helper = AudioteleAPIHelper()

    assignment_helper = AudioteleCallAssignmentHelper()
    registering_helper = AudioteleCallRegisteringHelper()

    @classmethod
    def from_settings(cls):
        return cls()

    def _parse_data(self, data):
        try:
            data = json.loads(data) if isinstance(data, str) else data
        except Exception:
            raise Exception('error parsing data')
        return data

    def process_call_start(self, raw_data):
        data = self._parse_data(raw_data)

        entries = data.get('data', None)

        if isinstance(entries, list):
            # it's a stub right now
            required_fields = ('time_enter', 'phone', 'agent', 'call_id')

            if not all(x in e for x in required_fields for e in entries):
                raise Exception('incorrect data format')

            processed_entries = [self.api_helper.make_call_enter_entry(**raw_entry) for raw_entry in entries]

            self.assignment_helper.register_call_entries(processed_entries)
            self.registering_helper.process_call_start(processed_entries)
        else:
            raise Exception('incorrect data format: no "data" field in request body')

    def process_call_finish(self, raw_data):
        data = self._parse_data(raw_data)

        entries = data.get('data', None)

        if isinstance(entries, list):
            # it's a stub right now
            required_fields = (
                'phone', 'agent', 'duration', 'is_answered', 'direction',
                'time_enter', 'time_connect', 'time_exit', 'call_id'
            )

            if not all(x in e for x in required_fields for e in entries):
                raise Exception('incorrect data format')

            processed_entries = []

            for raw_entry in entries:
                entry = self.api_helper.make_call_exit_entry(**raw_entry)
                if entry is not None:
                    processed_entries.append(entry)

            self.assignment_helper.remove_call_entries(processed_entries)
            self.registering_helper.process_call_end(processed_entries)
        else:
            raise Exception('incorrect data format: no "data" field in request body')
