import collections
import datetime
import logging

from django.db import transaction

from cars.core.util import datetime_helper, phone_number_helper
from cars.callcenter.core import StaffInfoHelper

from cars.request_aggregator.models import BeeperAgentEntry, BeeperCallEntry, BeeperCallTrackEntry


LOGGER = logging.getLogger(__name__)


class BeeperCallStatCollectingHelper(object):
    MAX_INTERNAL_CALL_ID_BULK = 512

    BeeperStatEntry = collections.namedtuple(
        'BeeperStatEntry',
        ('Direction', 'Time_enter', 'Time_connect', 'Time_exit', 'Duration', 'Connect_trial_count', 'is_answered',
         'Phone', 'Agent', 'Agent_login', 'Agent_print_name', 'RouterCallKeyDay', 'RouterCallKey', )
    )

    BeeperAgentUpdateEntry = collections.namedtuple(
        'BeeperAgentUpdateEntry', ('agent_login', 'agent_staff_login')
    )

    staff_info_helper = StaffInfoHelper.make_default()

    def __init__(self):
        self._agent_cache = {a.related_id: a for a in BeeperAgentEntry.objects.all()}

    def _get_agent_entry(self, agent_id, agent_print_name):
        if agent_id is None:
            return None

        if agent_id not in self._agent_cache:
            agent_entry = BeeperAgentEntry.objects.create(related_id=agent_id, related_name=agent_print_name)
            self._agent_cache[agent_id] = agent_entry

        return self._agent_cache[agent_id]

    def process_entries(self, entries):
        processed_entries = [self._process_entry(self._make_beeper_stat_entry(e)) for e in entries]
        filtered_entries = self._filter_entries(processed_entries)
        BeeperCallEntry.objects.bulk_create(filtered_entries)
        return len(processed_entries), len(filtered_entries)

    def _make_beeper_stat_entry(self, entry):
        flattened_entry = (entry.get(name, None) for name in self.BeeperStatEntry._fields)
        stat_entry = self.BeeperStatEntry._make(flattened_entry)
        return stat_entry

    def _process_entry(self, e):
        time_enter = datetime_helper.timestamp_to_datetime(e.Time_enter)

        time_connect = datetime_helper.timestamp_to_datetime(e.Time_connect) if e.Time_connect is not None else None

        if e.Time_exit is not None:
            time_exit = datetime_helper.timestamp_to_datetime(e.Time_exit)
        else:
            time_exit = time_enter + datetime.timedelta(seconds=e.Duration)

        duration = e.Duration if e.is_answered else None
        phone = phone_number_helper.normalize_phone_number(e.Phone)
        agent = self._get_agent_entry(e.Agent, e.Agent_print_name)

        if e.RouterCallKeyDay and e.RouterCallKey:
            external_call_id = '{}_{}'.format(e.RouterCallKeyDay, e.RouterCallKey)
        else:
            external_call_id = None

        processed_entry = BeeperCallEntry(
            direction=e.Direction,
            time_enter=time_enter,
            time_connect=time_connect,
            time_exit=time_exit,
            duration=duration,
            is_answered=e.is_answered,
            phone=phone,
            external_call_id=external_call_id,
            beeper_agent_entry_binding=agent,
        )

        return processed_entry

    def _filter_entries(self, entries):
        # simply one by one
        filtered_entries = [
            e for e in entries
            if not BeeperCallEntry.objects.filter(time_enter=e.time_enter, phone=e.phone).exists()
        ]
        return filtered_entries

    def update_agent_binding(self, entries):
        for e in entries:
            e = self.BeeperAgentUpdateEntry(**e)
            staff_entry = self.staff_info_helper.get_agent_entry(username=e.agent_staff_login)

            if staff_entry is not None:
                with transaction.atomic(savepoint=False):
                    agent_entries = BeeperAgentEntry.objects.select_for_update().filter(related_id=e.agent_login)

                    for agent_entry in agent_entries:
                        agent_entry.staff_entry_binding = staff_entry
                        agent_entry.save()

    def mark_duplicated_as_removed(self):
        all_tracks = BeeperCallTrackEntry.objects.all()

        track_mapping = collections.defaultdict(list)

        for t in all_tracks:
            if not t.file_attrs.get('to_remove', False) and t.raw_phone is not None:
                track_mapping[t.file_name].append(t.id)

        track_id_to_mark_as_removed = []

        for track_id_group in track_mapping.values():
            if len(track_id_group) > 1:
                track_id_to_mark_as_removed.extend(track_id_group[1:])

        with transaction.atomic(savepoint=False):
            for track_id in track_id_to_mark_as_removed:
                track = BeeperCallTrackEntry.objects.get(id=track_id)
                track.file_attrs['to_remove'] = True
                track.save()

        return len(track_id_to_mark_as_removed)

    def remove_duplicates(self):
        all_entries = BeeperCallEntry.objects.all()

        call_mapping = collections.defaultdict(list)

        for e in all_entries:
            _key = e.time_enter, str(e.phone)
            call_mapping[_key].append(e.id)

        call_id_to_be_removed = []

        for call_id_group in call_mapping.values():
            if len(call_id_group) > 1:
                call_id_to_be_removed.extend(call_id_group[1:])

        with transaction.atomic(savepoint=False):
            for call_id in call_id_to_be_removed:
                entry = BeeperCallEntry.objects.get(id=call_id)
                entry.delete()

        return call_id_to_be_removed

    def update_track_binding(self, offset=0, count=None, match_by_call_id=False, use_precise_time=True):
        if count is None:
            count = BeeperCallTrackEntry.objects.count()

        total_updated = idx = 0

        with transaction.atomic(savepoint=False):
            tracks_to_update = (
                BeeperCallTrackEntry.objects
                .select_for_update()
                .filter(call_entry__isnull=True)
                .order_by('id')
                [offset:offset + count]
            )

            for idx, t in enumerate(tracks_to_update):
                if t.file_attrs.get('to_remove', False):
                    continue

                filters = {}

                if match_by_call_id and t.call_id is not None:
                    filters['external_call_id'] = t.call_id
                else:
                    if t.phone is not None:
                        filters['phone'] = t.phone
                    else:
                        LOGGER.warning(
                            'bad formatted number: track id - {}, track file name - {}'.format(t.id, t.file_name)
                        )
                        continue

                    if use_precise_time:
                        filters['time_connect'] = t.time_id
                    else:
                        filters['time_enter__lte'] = t.time_id
                        filters['time_exit__gte'] = t.time_id

                call_entry = BeeperCallEntry.objects.filter(**filters).first()

                if call_entry is not None:
                    if not hasattr(call_entry, 'track'):
                        t.call_entry = call_entry
                        t.save()
                        total_updated += 1
                    else:
                        LOGGER.warning(
                            'duplicate found: track id - {}, track file name - {}, '
                            'found call entry id - {}, found track id - {}'
                            .format(t.id, t.file_name, call_entry.id, call_entry.track.id)
                        )

        total_performed = idx

        return total_performed, total_updated
