import enum
import logging

from django.urls import reverse

import cars.settings

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.core.mds.client import MDSClient, EncryptedMDSClient
from cars.core.util import datetime_helper
from cars.callcenter.core import StaffInfoHelper
from cars.settings import CALLCENTER as callcenter_settings, REQUEST_AGGREGATOR as request_aggregator_settings

from cars.request_aggregator.core.phone_binding_helper import UserProcessingHelper
from cars.request_aggregator.models.beeper_stats import BeeperCallEntry, BeeperCallTrackEntry, BeeperCallDirection
from cars.request_aggregator.serializers.common import AggregatedResponseEntry, RequestType

LOGGER = logging.getLogger(__name__)


class BeeperMDSHelper(object):
    def __init__(self, client, bucket_name):
        self._client = client
        self._bucket_name = bucket_name

    @classmethod
    def from_settings(cls):
        mds_authorization_extra_settings = callcenter_settings['mds']
        mds_client = MDSClient.from_settings(**mds_authorization_extra_settings)

        mds_connection_settings = request_aggregator_settings['beeper']['mds_connection']

        if mds_connection_settings['encrypt']:
            aes_key = mds_connection_settings['aes_key']
            mds_client = EncryptedMDSClient(mds_client, aes_key)

        bucket_name = mds_connection_settings['bucket_name']
        return cls(mds_client, bucket_name)

    def put_object(self, track_entry, file_content):
        assert isinstance(track_entry, BeeperCallTrackEntry)
        mds_file_key = track_entry.mds_key
        response = self._client.put_object(bucket=self._bucket_name, key=mds_file_key, body=file_content)
        return response

    def get_object(self, track_entry):
        assert isinstance(track_entry, BeeperCallTrackEntry)
        response = self._client.get_object(
            bucket=self._bucket_name, key=track_entry.mds_key
        )
        return response

    def get_object_content(self, track_entry):
        assert isinstance(track_entry, BeeperCallTrackEntry)
        content = self._client.get_object_content(
            bucket=self._bucket_name, key=track_entry.mds_key
        )
        return content


class BeeperCallStatus(enum.Enum):
    SERVICED = 'serviced'
    NOT_SERVICED = 'not_serviced'


class BeeperProcessingHelper(object):
    REQUEST_TYPE = RequestType.phone

    INCOMING_QUEUE_NAME = 'beeper_incoming'
    OUTGOING_QUEUE_NAME = 'beeper_outgoing'

    DATA_URL_TEMPLATE = 'https://carsharing.yandex-team.ru{api_url}?track_id={track_id}'

    staff_info_helper = StaffInfoHelper.make_default()
    saas_client = SaasDriveAdminClient.from_settings()
    mds_helper = BeeperMDSHelper.from_settings()
    user_processing_helper = UserProcessingHelper.from_settings()

    def __init__(self, call_track_view_actions, denied_operator_view_roles, denied_data_view_roles):
        self._call_track_view_actions = call_track_view_actions
        self._denied_operator_view_roles = denied_operator_view_roles
        self._denied_data_view_roles = denied_data_view_roles

    @classmethod
    def from_settings(cls):
        call_track_view_actions = request_aggregator_settings['callcenter']['call_track_view_actions']
        denied_operator_view_roles = request_aggregator_settings['callcenter']['denied_operator_view_roles']
        denied_data_view_roles = request_aggregator_settings['callcenter']['denied_data_view_roles']
        return cls(call_track_view_actions, denied_operator_view_roles, denied_data_view_roles)

    def check_request_type(self, request_type_collection=None):
        return not request_type_collection or self.REQUEST_TYPE.value in request_type_collection

    def get_call_url(self, *, track=None, track_id=None):
        call_url = None

        if track_id is None and track is not None:
            track_id = track.id

        if track_id is not None:
            api_url = reverse('request_aggregator:beeper-tracks')
            call_url = self.DATA_URL_TEMPLATE.format(api_url=api_url, track_id=track_id)

        return call_url

    def get_requests(self, request=None, limit=None, **kwargs):
        if self._check_phone_not_matched(**kwargs):
            return []

        filters = self._make_call_filters(**kwargs)

        if not filters and limit is None:
            raise Exception('at least one filter must be provided')

        call_entries = (
            BeeperCallEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(**filters)
            .select_related('track', 'beeper_agent_entry_binding__staff_entry_binding')
            .order_by('-time_enter')
            [:limit]
        )

        users_mapping = self._make_call_users_mapping(call_entries, request, kwargs)

        access_mapping = self._make_access_mapping(request)

        response_entries = self._transform_raw_entries(call_entries, users_mapping, access_mapping)

        return response_entries

    def _check_phone_not_matched(self, user=None, phone_number=None, **kwargs):
        # user phone number is not specified neither explicitly nor present in user info
        return user is not None and phone_number is None

    def _make_call_filters(self, phone_number=None, since=None, until=None, staff_entry_binding=None, **kwargs):
        filters = {}

        if phone_number is not None:
            filters['phone'] = phone_number

        if since is not None:
            filters['time_enter__gte'] = since

        if until is not None:
            filters['time_enter__lt'] = until

        if staff_entry_binding is not None:
            filters['beeper_agent_entry_binding__staff_entry_binding'] = staff_entry_binding

        return filters

    def _make_call_users_mapping(self, call_entries, request, kwargs):
        if 'user' in kwargs:
            raw_entry_phone_bindings = {None: kwargs['user']}
            self.user_processing_helper.filter_deleted_users(raw_entry_phone_bindings, request)

            entry_phone_bindings = self.user_processing_helper.format_users(
                default=raw_entry_phone_bindings[None], update_from_key='phone'
            )

        else:
            raw_entry_phone_bindings = self.user_processing_helper.get_entries_phone_bindings(call_entries)
            self.user_processing_helper.filter_deleted_users(raw_entry_phone_bindings, request)

            entry_phone_bindings = self.user_processing_helper.format_users(
                raw_entry_phone_bindings, update_from_key='phone'
            )

        users_mapping = {
            call_entry.id: entry_phone_bindings[call_entry.phone]
            for call_entry in call_entries
        }

        return users_mapping

    def _transform_raw_entries(self, call_entries, users_mapping, access_mapping):
        response_entries_as_mappings = [
            self._process_call_entry(entry, users_mapping, access_mapping)._asdict()
            for entry in call_entries
        ]
        return response_entries_as_mappings

    def _process_call_entry(self, entry, users_mapping, access_mapping):
        if entry.direction == BeeperCallDirection.INCOMING.value:
            queue = self.INCOMING_QUEUE_NAME
        else:
            queue = self.OUTGOING_QUEUE_NAME

        time_enter = entry.time_enter.timestamp()
        time_exit = entry.time_exit.timestamp()

        is_answered = entry.is_answered
        status = BeeperCallStatus.SERVICED if is_answered else BeeperCallStatus.NOT_SERVICED
        time_connect = entry.time_connect.timestamp() if is_answered else None

        duration = entry.duration
        duration_print = datetime_helper.duration_to_str(duration)

        agent_info_entry = entry.beeper_agent_entry_binding

        if agent_info_entry is not None:
            staff_binding = agent_info_entry.staff_entry_binding

            if staff_binding is not None:
                agent = self.staff_info_helper.format_staff_entry(staff_binding)
            else:
                agent = self.staff_info_helper.format_entry_generic(
                    print_name=(agent_info_entry.related_name or '')
                )
        else:
            agent = self.staff_info_helper.format_staff_entry(None)

        if access_mapping['has_access_to_operators']:
            operators = [agent]
        else:
            operators = []

        phone = str(entry.phone) if entry.phone is not None else entry.phone

        if access_mapping['has_access_to_call_id']:
            data_url = self.get_call_url(track=getattr(entry, 'track', None))
        else:
            data_url = None

        data = {'track_url': data_url}

        request_description = {
            'id': None,
            'origin': None,
            'queue': queue,
            'type': self.REQUEST_TYPE.value,
        }

        user = users_mapping.get(entry.id, None)

        response_entry = AggregatedResponseEntry(
            time_enter=time_enter, time_connect=time_connect, time_exit=time_exit,
            connect_trial_count=1, duration=duration, duration_print=duration_print, status=status.value,
            phone=phone, operators=operators, tags=[], request=request_description, user=user,
            data_url=data_url, data=data, message=None
        )

        return response_entry

    def _make_access_mapping(self, request=None):
        access_mapping = {}

        if request is not None:
            has_access_to_call_id = (
                self.saas_client.check_user_action(
                    *self._call_track_view_actions, request=request, require_all=False
                ) and
                not self.saas_client.check_user_role(
                    *self._denied_data_view_roles, request=request, require_all=False
                )
            )
            has_access_to_operators = not self.saas_client.check_user_role(
                *self._denied_operator_view_roles, request=request, require_all=False
            )
        else:
            has_access_to_call_id = False
            has_access_to_operators = False

        access_mapping['has_access_to_call_id'] = has_access_to_call_id
        access_mapping['has_access_to_operators'] = has_access_to_operators

        return access_mapping

    def get_call_track(self, *, track=None, track_id=None, external_call_id=None):
        assert (track is not None) or (track_id is not None) or (external_call_id is not None), 'either track or track_id or call id must be specified'

        if track is None:
            if track_id is not None:
                track = (
                    BeeperCallTrackEntry.objects.using(cars.settings.DB_RO_ID)
                    .filter(id=track_id).first()
                )
            elif external_call_id is not None:
                call_ids = list(
                    BeeperCallEntry.objects.using(cars.settings.DB_RO_ID)
                    .filter(external_call_id=external_call_id).values_list('id', flat=True)
                )
                if call_ids:
                    track = (
                        BeeperCallTrackEntry.objects.using(cars.settings.DB_RO_ID)
                        .filter(call_entry_id__in=call_ids).first()
                    )
            else:
                assert False, 'invalid track id parameters'

        data = None
        if track is not None:
            data = self.mds_helper.get_object_content(track_entry=track)

        return data
