from collections import namedtuple
import datetime
import enum
from itertools import chain
import logging
import re

from urllib.parse import urlencode

import pytz
import requests

from cars.core.telephony import TelephonyHelperBase
from cars.core.util import phone_number_helper

from cars.request_aggregator.models.chat2desk_stats import Chat2DeskMessageEntry, Chat2DeskEntryType

LOGGER = logging.getLogger(__name__)


class Chat2DeskApiHelper(TelephonyHelperBase):
    MAX_OPERATORS_BATCH_SIZE = 25
    MAX_MESSAGES_BATCH_SIZE = 200
    MAX_CLIENTS_BATCH_SIZE = 100

    CLIENT_NAME_RE = re.compile('[А-Яа-яЁё\- ]+')

    MOCKED_CLIENT_EN_INFO = 'dummy client'
    MOCKED_CLIENT_RU_INFO = 'несуществующий клиент'
    MOCKED_CLIENT_IDENTIFICATION_INFO = '{}---{}'.format(MOCKED_CLIENT_EN_INFO, MOCKED_CLIENT_RU_INFO)

    SpecificClientInfo = namedtuple('SpecificClientInfo', ['client_id', 'phones', 'extra_info'])
    DialogMetaInfo = namedtuple('DialogMetaInfo', ['client_id', 'start', 'end', 'related_id', 'transport'])

    class ApiStatus(enum.Enum):
        SUCCESS = 'success'
        FAIL = 'fail'

    def __init__(self, api_token, request_timeout=15, retries=3):
        super().__init__(request_timeout=request_timeout, retries=retries)

        self._base_url = 'https://api.chat2desk.com'
        self._section_urls = {
            'messages': '/v1/messages',
            'clients': '/v1/clients',
            'dialogs': '/v1/dialogs',
            'operators': '/v1/operators',
        }

        self._setup_authorization_headers(api_token)

    def _setup_authorization_headers(self, api_token):
        self._session.headers.update({'Authorization': '{}'.format(api_token)})

    @classmethod
    def parse_chat2desk_datetime_format(cls, value):
        return pytz.UTC.localize(datetime.datetime.strptime(value, '%Y-%m-%dT%H:%M:%S UTC'))

    def _try_extract_client_name(self, text):
        return map(lambda x: x.strip(), self.CLIENT_NAME_RE.findall(text))

    def _perform_request(self, api_section, *extra_sections, raise_for_status=False, **params):
        base_api_url = self._get_api_url(api_section)
        api_url = '/'.join(chain((base_api_url, ), map(str, extra_sections)))
        parameterized_api_url = '{}?{}'.format(api_url, urlencode(params))

        raw_data = super()._perform_request(parameterized_api_url, raise_for_status=raise_for_status)
        data, meta = self._try_parse_response(raw_data)

        return data, meta

    def _get_api_url(self, section):
        return self._base_url + self._section_urls[section]

    def _try_parse_response(self, raw_data):
        data, meta = None, {}

        if raw_data is not None and raw_data['status'] == self.ApiStatus.SUCCESS.value:
            data = raw_data['data']
            meta = raw_data.get('meta', None)

        return data, meta

    def get_messages(self, offset=0, limit=20, client_id=None):
        limit = min(limit, self.MAX_MESSAGES_BATCH_SIZE)
        params = {'offset': offset, 'limit': limit, }
        if client_id is not None:
            params['client_id'] = client_id
        data, _ = self._perform_request('messages', **params)
        return data

    def get_remaining_messages(self, total_saved, client_id=None):
        entries = []

        total_available = self.get_total_message_count(client_id)

        if total_available is not None:
            request_limit = total_available - total_saved
            if request_limit:
                entries = self.get_messages(total_saved, request_limit, client_id)

        return entries, total_available

    def get_total_message_count(self, client_id=None):
        params = {'client_id': client_id} if client_id is not None else {}
        _, meta = self._perform_request('messages', **params)
        return meta.get('total', None)

    def get_operators(self, offset=0, limit=20):
        limit = min(limit, self.MAX_OPERATORS_BATCH_SIZE)
        data, _ = self._perform_request('operators', offset=offset, limit=limit)
        return data

    def get_remaining_operators(self, total_saved):
        entries = []

        total_available = self.get_total_operators_count()

        if total_available is not None:
            request_limit = total_available - total_saved
            if request_limit:
                entries = self.get_operators(total_saved, request_limit)

        return entries, total_available

    def get_total_operators_count(self):
        _, meta = self._perform_request('operators')
        return meta.get('total', None)

    def get_client(self, client_id):
        try:
            data, _ = self._perform_request('clients', client_id, raise_for_status=True)
        except requests.HTTPError as exc:
            if exc.response.status_code == 404:
                # try to avoid API bug with missed clients
                data = self._make_client_mock(client_id)
            else:
                data = None
        return data

    def _make_client_mock(self, client_id):
        mocked_client_info = {
            'id': client_id,
            'comment': self.MOCKED_CLIENT_IDENTIFICATION_INFO,
            'name': '',
            'assigned_name': None,
            'phone': '',
            'client_phone': None,
        }
        return mocked_client_info

    def get_clients(self, offset=0, limit=20):
        limit = min(limit, self.MAX_CLIENTS_BATCH_SIZE)
        data, _ = self._perform_request('clients', offset=offset, limit=limit)
        return data

    def get_remaining_clients(self, total_saved):
        entries = []

        total_available = self.get_total_clients_count()

        if total_available is not None:
            request_limit = total_available - total_saved
            if request_limit:
                entries = self.get_clients(total_saved, request_limit)

        return entries, total_available

    def get_total_clients_count(self):
        _, meta = self._perform_request('clients')
        return meta.get('total', None)

    def get_client_dialogs(self, client_id):
        data, _ = self._perform_request('clients', client_id, 'dialogs')
        return data

    def get_client_transports(self, client_id):
        data, _ = self._perform_request('clients', client_id, 'transport')
        return data

    def get_dialog(self, dialog_id):
        data, _ = self._perform_request('dialogs', dialog_id)
        return data

    def get_client_dialog_info(self, client_id, dialog_id=None, transport=None):
        entry = None

        dialogs = self.get_client_dialogs(client_id) if dialog_id is None else (self.get_dialog(dialog_id), )
        transports = self.get_client_transports(client_id) if transport is None else ({'transports': [transport]}, )

        if dialogs and transports:
            available_transports = list(chain(*[t['transports'] for t in transports]))

            # client is expected to have only one dialog and one transport type,
            #  as no one is matched by any criteria
            if len(dialogs) == 1 and len(available_transports) == 1:
                (dialog, ) = dialogs
                (transport, ) = available_transports
                entry = self.DialogMetaInfo(
                    client_id=client_id,
                    start=self.parse_chat2desk_datetime_format(dialog['begin']),
                    end=self.parse_chat2desk_datetime_format(dialog['end']) if dialog['end'] else None,
                    related_id=int(dialog['id']),
                    transport=transport,
                )
            else:
                LOGGER.error('cannot process dialog entry due to unexpected format')

        return entry

    def get_specific_client_info(self, client_id, raw_client=None):
        raw_client = self.get_client(client_id) if raw_client is None else raw_client

        if raw_client is None:
            return None

        possible_phones = []
        possible_extra_id_info = []

        possible_phones.append(phone_number_helper.normalize_phone_number(raw_client['client_phone']))

        extra_fields = [raw_client['name'], raw_client['assigned_name'], raw_client.get('comment', None), ]

        client_id_info = raw_client['phone']
        if '[tg]' in client_id_info:  # a telegram user
            possible_extra_id_info.append(client_id_info)
        else:
            extra_fields.append(client_id_info)

        for v in filter(None, extra_fields):
            normalized_phone = phone_number_helper.normalize_phone_number(v)
            if normalized_phone:
                possible_phones.append(normalized_phone)
            else:
                possible_extra_id_info.extend(self._try_extract_client_name(v))

        phones = list({p for p in possible_phones if p})
        extra_id_info = ';'.join({e for e in possible_extra_id_info if e})

        return self.SpecificClientInfo(client_id=client_id, phones=phones, extra_info=extra_id_info)

    def check_message_request_end(self, message):
        closed_by_operator = (message['type'] == Chat2DeskEntryType.SYSTEM.value
                              and message['text'] and message['text'].startswith('Диалог был закрыт'))
        closed_by_client = (message['type'] == Chat2DeskEntryType.FROM_CLIENT.value
                            and message['text'] and message['text'].startswith('Завершить чат'))
        return closed_by_operator or closed_by_client

    def check_prev_message_request_end(self, message, local_messages=()):
        client_id = message['client_id']

        specific_local_messages = [m for m in local_messages if m.related_client.related_id == client_id]

        if specific_local_messages:
            last_client_message = specific_local_messages[-1]
        else:
            last_client_message = Chat2DeskMessageEntry.objects.filter(
                related_id__lt=message['id'],
                related_client__related_id=client_id,
            ).order_by('-related_id').first()

        return (last_client_message is None or
                last_client_message.entry_type == Chat2DeskEntryType.REQUEST_END.value)
