import logging

from django.db import transaction
from django.db.models import Max, Q

from cars.core.util import phone_number_helper
from cars.callcenter.core import StaffInfoHelper
from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.models.chat2desk_stats import (
    Chat2DeskOperatorEntry, Chat2DeskClientEntry, Chat2DeskMessageEntry, Chat2DeskDialogEntry,
    Chat2DeskAttachmentType, Chat2DeskEntryType
)

from .api_helper import Chat2DeskApiHelper

LOGGER = logging.getLogger(__name__)


class Chat2DeskCollectingHelperBase(object):
    staff_info_helper = StaffInfoHelper.make_default()

    def __init__(self, api_token, request_timeout=15, retries=3):
        self._api_helper = Chat2DeskApiHelper(api_token, request_timeout=request_timeout, retries=retries)

    def _process_operator_entry(self, operator_entry):
        operator_id = operator_entry['id']

        print_name = ' '.join(filter(None, (operator_entry['first_name'], operator_entry['last_name'])))
        phone = (operator_entry['phone'] or None)  # None instead of blank
        email = operator_entry['email']

        staff_entry = self.staff_info_helper.get_agent_entry(email=email)

        if staff_entry is not None:
            related_staff_work_phone = staff_entry.work_phone
        else:
            LOGGER.warning('no staff entry with email {} found for operator {}'.format(email, operator_id))
            related_staff_work_phone = None

        return Chat2DeskOperatorEntry(
            related_id=operator_id,
            related_staff_work_phone=related_staff_work_phone,
            print_name=print_name,
            phone=phone,
            email=email,
            role=operator_entry['role'],
            staff_entry_binding=staff_entry,
        )

    def _process_client_entry(self, client_id, dialog_info, *, raw_client=None):
        client_info = self._api_helper.get_specific_client_info(client_id, raw_client=raw_client)

        if client_info is None:
            LOGGER.error('no information found about requested client with id {}'.format(client_id))
            return None

        time_id = dialog_info.start if dialog_info else None

        client_phones = client_info.phones or [None]  # create the first entry without phone

        yield from (
            Chat2DeskClientEntry(
                related_id=client_id,
                time_id=time_id,
                assigned_phone=phone,
                extra_identification_info=client_info.extra_info,
            ) for phone in client_phones
        )

    def _process_dialog_entry(self, client_entry, dialog_info):
        yield Chat2DeskDialogEntry(
            related_client=client_entry,
            related_id=dialog_info.related_id,
            transport=dialog_info.transport,
        )

    def _process_message_entry(self, message_entry, related_clients=None, related_dialog=None):
        related_client_id = message_entry['client_id']

        if related_clients is None:
            related_clients = list(
                Chat2DeskClientEntry.objects.filter(related_id=related_client_id)
                .order_by('id')
            )

            if not related_clients:
                related_clients = self._request_client_update(message_entry)

        transport = message_entry['transport']

        related_dialog_id = message_entry['dialog_id']

        if related_dialog is None and related_dialog_id is not None:
            related_dialog = Chat2DeskDialogEntry.objects.filter(related_id=related_dialog_id).first()

            if related_dialog is None:
                related_dialog = self._request_dialog_update(message_entry, related_clients)

        if self._is_extra_update_required(message_entry, related_clients, related_dialog):
            return None

        registration_client_entry = related_clients[0] if related_clients else None

        operator_id = message_entry['operator_id']
        related_operator = None

        if operator_id is not None:
            related_operator = Chat2DeskOperatorEntry.objects.filter(related_id=operator_id).first()

            if related_operator is None:
                related_operator = self._request_operator_update(message_entry)

        related_id = message_entry['message_id']
        related_request_id = message_entry.get('request_id', None)

        time_id = message_entry['time_id']

        entry_type = message_entry['type']

        text = message_entry['text'] or ''

        if text is not None:
            text = text[:Chat2DeskMessageEntry.MAX_TEXT_LENGTH]
            self._try_update_client_info(message_entry, time_id, registration_client_entry, related_clients)

        attachment_type = None
        attachment_url = None

        # there could be both text and attachment provided by the web hook
        for field in Chat2DeskAttachmentType:
            value = message_entry.get(field.value, None)

            if value is not None:
                attachment_type = field.value
                attachment_url = value

        return Chat2DeskMessageEntry(
            related_id=related_id,
            related_request_id=related_request_id,
            related_dialog_id=related_dialog_id,
            time_id=time_id,
            related_client=registration_client_entry,
            related_operator=related_operator,
            text=text,
            attachment_url=attachment_url,
            attachment_type=attachment_type,
            transport=transport,
            entry_type=entry_type,
        )

    def _request_client_update(self, message_entry):
        return []

    def _request_dialog_update(self, message_entry, client_entries):
        return None

    def _request_operator_update(self, message_entry):
        return None

    def _is_extra_update_required(self, message_entry, related_clients, related_dialog):
        return False

    def _try_update_client_info(self, message_entry, time_id, related_client, all_related_clients):
        if message_entry['type'] not in (
                Chat2DeskEntryType.REQUEST_START.value,
                Chat2DeskEntryType.FROM_CLIENT.value,
                Chat2DeskEntryType.SYSTEM.value
        ):
            return

        if related_client is None:
            return

        related_id = related_client.related_id
        extra_identification_info = related_client.extra_identification_info

        message_text = message_entry['text'] or ''
        possible_phones = phone_number_helper.iter_all_possible_phone_numbers(message_text)

        assigned_client_phones = [str(c.assigned_phone) for c in all_related_clients if c.assigned_phone]

        entries = [
            Chat2DeskClientEntry(
                related_id=related_id,
                time_id=time_id,
                assigned_phone=phone,
                extra_identification_info=extra_identification_info,
            )
            for phone in possible_phones
            if phone not in assigned_client_phones
        ]

        if entries:
            Chat2DeskClientEntry.objects.bulk_create(entries)
            LOGGER.info('chat2desk clients: {} extra entries created for client {}, assigned phones - {}'
                        .format(len(entries), related_id, [str(e.assigned_phone) for e in entries]))


class Chat2DeskCollectingHelper(Chat2DeskCollectingHelperBase):
    def __init__(self, api_token, request_timeout=15, retries=3):
        super().__init__(api_token, request_timeout, retries)
        self._last_total_processed_clients_count = None
        self._force_client_dialog_mapping = {}  # pairs (client_id: (dialog_id, transport))

    @classmethod
    def from_settings(cls):
        chat2desk_settings = settings['chat2desk']

        api_token = chat2desk_settings['api_token']
        request_timeout = chat2desk_settings['request_timeout']

        return cls(api_token, request_timeout)

    def update_data(self):
        processed_operators_count = self._update_operators()
        processed_clients_count = self._update_clients()
        processed_messages_count = self._update_messages()
        return processed_operators_count, processed_clients_count, processed_messages_count

    def _update_operators(self):
        total_saved_operators = Chat2DeskOperatorEntry.objects.all().count()

        raw_operator_entries, total_available = self._api_helper.get_remaining_operators(total_saved_operators)

        if raw_operator_entries is None:
            LOGGER.error('cannot obtain operators using chat2desk API')
            raw_operator_entries = []

        processed_entries = [self._process_operator_entry(e) for e in raw_operator_entries]

        Chat2DeskOperatorEntry.objects.bulk_create(processed_entries)

        processed_operators_count = len(processed_entries)

        LOGGER.info('chat2desk operators: already saved - {}, processed - {}, available - {}'
                    .format(total_saved_operators, processed_operators_count, total_available))

        return processed_operators_count

    def _update_clients(self):
        shared_dialog_info_mapping = {}

        total_processed_clients_count = Chat2DeskClientEntry.objects.distinct('related_id').count()
        raw_client_entries, total_available = self._api_helper.get_remaining_clients(total_processed_clients_count)

        if raw_client_entries is None:
            LOGGER.error('cannot obtain clients using chat2desk API')
            raw_client_entries = []

        client_dialog_mapping = {e['id']: (None, None) for e in raw_client_entries}

        extra_mapping = self._get_extra_client_dialog_mapping(client_dialog_mapping, total_processed_clients_count)

        client_dialog_mapping.update(extra_mapping)

        with transaction.atomic(savepoint=False):
            processed_client_entries = self._process_clients(client_dialog_mapping, shared_dialog_info_mapping)
            processed_dialog_entries = self._process_dialogs(client_dialog_mapping, shared_dialog_info_mapping)

        if extra_mapping:
            LOGGER.info('removing extra client id entries: {}'.format(list(extra_mapping.keys())))
            for extra_cid in extra_mapping:
                self._force_client_dialog_mapping.pop(extra_cid)

        self._last_total_processed_clients_count = total_processed_clients_count

        processed_clients_count = len(processed_client_entries)
        processed_dialogs_count = len(processed_dialog_entries)

        LOGGER.info('chat2desk clients: already saved - {}, processed - {}, dialogs processed - {}, available - {}'
                    .format(total_processed_clients_count, processed_clients_count,
                            processed_dialogs_count, total_available))

        return processed_clients_count

    def _get_extra_client_dialog_mapping(self, client_dialog_mapping, total_processed_clients_count):
        # try to determine new client obtaining process stuck
        is_stuck_determined = (self._last_total_processed_clients_count is not None
                               and self._last_total_processed_clients_count == total_processed_clients_count)

        if not (client_dialog_mapping and is_stuck_determined):
            # try to maintain range contiguous by adding only missed or already processed entries
            max_related_id = self._get_max_client_related_id(client_dialog_mapping, total_processed_clients_count)

            LOGGER.info('force client filter applied on ids less or equal to {}'.format(max_related_id))

            extra_client_dialog_mapping = {cid: data for cid, data in self._force_client_dialog_mapping.items()
                                           if max_related_id is None or cid <= max_related_id}
        else:
            LOGGER.info('no force client filter applied: either no new clients available'
                        ' or stuck has been determined')
            extra_client_dialog_mapping = self._force_client_dialog_mapping.copy()

        return extra_client_dialog_mapping

    def _get_max_client_related_id(self, client_dialog_mapping, total_processed_clients_count):
        max_actual_related_id = max(client_dialog_mapping) if client_dialog_mapping else None

        if total_processed_clients_count:
            max_saved_related_id = Chat2DeskClientEntry.objects.aggregate(Max('related_id'))['related_id__max']
        else:
            max_saved_related_id = None

        max_related_id = max(filter(None, (max_actual_related_id, max_saved_related_id)), default=None)
        return max_related_id

    def _process_clients(self, related_client_dialog_mapping, dialog_info_mapping):
        existing_clients, _ = self._get_existing_related_entries(related_client_dialog_mapping)

        existing_client_ids = {e.related_id for e in existing_clients}

        processed_client_entries = []

        for cid, (dialog_id, transport) in related_client_dialog_mapping.items():
            if cid not in existing_client_ids:
                if cid not in dialog_info_mapping:
                    dialog_info_mapping[cid] = self._api_helper.get_client_dialog_info(cid, dialog_id, transport)

                dialog_info = dialog_info_mapping[cid]
                client_entries = self._process_client_entry(cid, dialog_info)
                processed_client_entries.extend(client_entries)

        Chat2DeskClientEntry.objects.bulk_create(processed_client_entries)

        return processed_client_entries

    def _process_dialogs(self, related_client_dialog_mapping, dialog_info_mapping):
        # get newly created instances, ones obtained before cannot be re-used
        existing_clients, existing_dialogs = self._get_existing_related_entries(related_client_dialog_mapping)

        existing_dialog_ids = {e.related_id for e in existing_dialogs}

        processed_dialog_entries = []

        for client_entry in existing_clients:
            cid = client_entry.related_id

            if cid not in dialog_info_mapping:
                dialog_id, transport = related_client_dialog_mapping.get(cid, (None, None))
                dialog_info_mapping[cid] = self._api_helper.get_client_dialog_info(cid, dialog_id, transport)

            dialog_info = dialog_info_mapping[cid]

            if dialog_info is not None and dialog_info.related_id not in existing_dialog_ids:
                dialog_entries = self._process_dialog_entry(client_entry, dialog_info)
                processed_dialog_entries.extend(dialog_entries)

        Chat2DeskDialogEntry.objects.bulk_create(processed_dialog_entries)

        return processed_dialog_entries

    def _get_existing_related_entries(self, related_client_dialog_mapping):
        related_client_ids = related_client_dialog_mapping.keys()
        existing_client_entries = Chat2DeskClientEntry.objects.filter(
            related_id__in=related_client_ids
        ).order_by('related_id', 'time_id').distinct('related_id')

        related_dialog_ids = {dialog_id for dialog_id, transport in related_client_dialog_mapping.values()}
        existing_dialog_entries = Chat2DeskDialogEntry.objects.filter(
            Q(related_client_id__in=existing_client_entries) | Q(related_id__in=related_dialog_ids)
        )

        return existing_client_entries, existing_dialog_entries

    def _is_extra_update_required(self, message_entry, related_clients, related_dialog):
        require_client_update = (not related_clients)
        require_dialog_update = message_entry['dialog_id'] is not None and related_dialog is None

        is_extra_update_required = (require_client_update or require_dialog_update)

        if is_extra_update_required:
            # cannot process further messages till clients are created and (or) dialogs are updated
            related_client_id = message_entry['client_id']
            transport = message_entry['transport']
            related_dialog_id = message_entry['dialog_id']

            LOGGER.info('force update of client id {} and client dialog id {} requested'
                        .format(related_client_id, related_dialog_id))
            self._force_client_dialog_mapping.setdefault(related_client_id, (related_dialog_id, transport))

        return is_extra_update_required

    def _update_messages(self):
        total_saved_messages = Chat2DeskMessageEntry.objects.all().count()
        raw_messages_entries, total_available = self._api_helper.get_remaining_messages(total_saved_messages)

        if raw_messages_entries is None:
            LOGGER.error('cannot obtain messages using chat2desk API')
            raw_messages_entries = []

        processed_entries = []
        existing_message_ids = self._get_existing_message_ids(raw_messages_entries) if raw_messages_entries else ()

        for e in raw_messages_entries:
            if e['id'] in existing_message_ids:
                continue

            # dialog part may be stored locally and not committed yet
            e['type'] = self._get_message_entry_type(e, processed_entries)
            processed_entry = self._process_message_entry(e)

            if processed_entry is not None:
                processed_entries.append(processed_entry)
            else:
                break

        Chat2DeskMessageEntry.objects.bulk_create(processed_entries)

        processed_messages_count = len(processed_entries)

        LOGGER.info('chat2desk messages: already saved - {}, processed - {}, available - {}'
                    .format(total_saved_messages, processed_messages_count, total_available))

        return processed_messages_count

    def _get_existing_message_ids(self, raw_messages_entries):
        # intended to use in parallel with webhook
        message_ids = (e['id'] for e in raw_messages_entries)
        existing_message_ids = set(
            Chat2DeskMessageEntry.objects.filter(related_id__in=message_ids)
            .values_list('related_id', flat=True)
        )
        return existing_message_ids

    def _process_message_entry(self, message_entry, related_clients=None, related_dialog=None):
        message_entry['message_id'] = message_entry['id']
        message_entry['time_id'] = self._api_helper.parse_chat2desk_datetime_format(message_entry['created'])
        return super()._process_message_entry(message_entry, related_clients, related_dialog)

    def _get_message_entry_type(self, message_entry, local_messages):
        if message_entry['type'] is None:
            entry_type = Chat2DeskEntryType.SYSTEM.value  # e.g. internal comments
        elif self._api_helper.check_message_request_end(message_entry):
            entry_type = Chat2DeskEntryType.REQUEST_END.value
        elif (message_entry['type'] == Chat2DeskEntryType.FROM_CLIENT.value and
              self._api_helper.check_prev_message_request_end(message_entry, local_messages)):
            entry_type = Chat2DeskEntryType.REQUEST_START.value
        else:
            entry_type = message_entry['type']

        return entry_type
