import logging

import cars.settings

from cars.request_aggregator.core.common_helper import YTHelper
from cars.request_aggregator.core.request_time_sync_helper import RequestEntryCountSyncHelper

from cars.request_aggregator.models.call_center_common import SyncOrigin
from cars.request_aggregator.models.chat2desk_stats import (
    Chat2DeskMessageEntry, Chat2DeskDialogEntry, Chat2DeskOperatorEntry, Chat2DeskClientEntry
)


LOGGER = logging.getLogger(__name__)


class Chat2deskExportSyncingHelper(RequestEntryCountSyncHelper):
    @classmethod
    def from_settings(cls):
        return cls(stat_sync_origin=SyncOrigin.CHAT2DESK_CHATS_EXPORTING)


class Chat2deskExportProcessingHelper(object):
    DATA_URL_TEMPLATE = 'https://web.chat2desk.com/chat/all?dialogID={dialog_id}'

    def __init__(self):
        self._dialog_cache = None
        self._operator_cache = None
        self._client_cache = None

    def get_dialog_cache(self):
        dialogs = (
            Chat2DeskDialogEntry.objects
            .using(cars.settings.DB_RO_ID)
            .all()
            .values('id', 'transport', 'related_id')
        )
        dialog_cache = {(x['id'], x['transport']): x['related_id'] for x in dialogs}
        return dialog_cache

    def get_operator_cache(self):
        operators = Chat2DeskOperatorEntry.objects.using(cars.settings.DB_RO_ID).all().values('id', 'email')
        operator_cache = {x['id']: x['email'].split('@')[0] for x in operators}
        return operator_cache

    def get_client_cache(self):
        clients = Chat2DeskClientEntry.objects.using(cars.settings.DB_RO_ID).all().values('id', 'related_id')
        client_cache = {x['id']: x['related_id'] for x in clients}
        return client_cache

    def init_cache(self, force=True):
        if self._dialog_cache is None or force:
            self._dialog_cache = self.get_dialog_cache()

        if self._operator_cache is None or force:
            self._operator_cache = self.get_operator_cache()

        if self._client_cache is None or force:
            self._client_cache = self.get_client_cache()

    def get_dialog_id(self, client_id, transport):
        return self._dialog_cache.get((client_id, transport))

    def get_dialog_url(self, dialog_id):
        return self.DATA_URL_TEMPLATE.format(dialog_id=dialog_id) if dialog_id is not None else ''

    def get_operator_login(self, operator_id):
        return self._operator_cache.get(operator_id)

    def get_external_client_id(self, client_id):
        return self._client_cache.get(client_id)


class Chat2deskExportHelper(object):
    CHUNKS_MERGE_LIMIT = 500

    def __init__(self, yt_export_table):
        self._yt_helper = YTHelper(yt_export_table)
        self._sync_helper = Chat2deskExportSyncingHelper.from_settings()

        self._processing_helper = Chat2deskExportProcessingHelper()
        self._processing_helper.init_cache()

    @classmethod
    def from_settings(cls):
        return cls(cars.settings.REQUEST_AGGREGATOR['chat2desk']['export_table'])

    def export(self):
        total_count = Chat2DeskMessageEntry.objects.count()

        for idx, (start_offset, end_offset) in enumerate(self._sync_helper.iter_indexes_to_process(total_count)):
            data = self._iter_data_to_store(start_offset, end_offset)
            self._yt_helper.export_data(data)

            if not (idx + 1) % self.CHUNKS_MERGE_LIMIT:
                self.optimize()

    def _iter_data_to_store(self, start_offset, end_offset):
        message = None

        messages = Chat2DeskMessageEntry.objects.using(cars.settings.DB_RO_ID).order_by('id')[start_offset:end_offset]

        try:
            for message in messages:
                yield self._process_message(message)
        except Exception:
            message_id = getattr(message, 'id', None)
            LOGGER.exception('error processing message with id {}'.format(message_id))
            raise

    def _process_message(self, message):
        operator_username = self._processing_helper.get_operator_login(message.related_operator_id)

        data = message.text

        if message.attachment_url is not None:
            if data:
                data += '\n'

            data += str(message.attachment_url)

        client_id = self._processing_helper.get_external_client_id(message.related_client_id)

        dialog_id = message.related_dialog_id

        if dialog_id is None:
            dialog_id = self._processing_helper.get_dialog_id(message.related_client_id, message.transport)

        dialog_url = self._processing_helper.get_dialog_url(dialog_id)

        formatted_message = {
            'operator_username': operator_username,
            'datetime': str(message.time_id),
            'timestamp': message.time_id.timestamp(),
            'message_type': message.entry_type,
            'data': data,
            'client_id': client_id,
            'dialog_id': dialog_id,
            'dialog_url': dialog_url,
            'transport': message.transport,
        }
        return formatted_message

    def optimize(self):
        self._yt_helper.merge_chunks()
