import datetime
import itertools
import logging

import cars.settings

from cars.core.util import datetime_helper
from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.callcenter.core import StaffInfoHelper

from cars.request_aggregator.core.common_helper import collection_to_mapping
from cars.request_aggregator.core.phone_binding_helper import UserProcessingHelper
from cars.request_aggregator.models.chat2desk_stats import (
    Chat2DeskClientEntry, Chat2DeskMessageEntry, Chat2DeskDialogEntry, Chat2DeskEntryType
)
from cars.request_aggregator.models.call_tags import RequestOriginType
from cars.request_aggregator.serializers.common import AggregatedResponseEntry, RequestType


LOGGER = logging.getLogger(__name__)


class Chat2DeskProcessingHelper(object):
    REQUEST_TYPE = RequestType.messenger

    DATA_URL_TEMPLATE = 'https://web.chat2desk.com/chat/all?dialogID={dialog_id}'

    MAX_REQUEST_END_GAP = 60 * 60 * 6  # up to 6 hours
    ESTIMATED_DIALOG_SIZE = 50  # max average number of entries in a dialog

    saas_client = SaasDriveAdminClient.from_settings()
    staff_info_helper = StaffInfoHelper.make_default()
    user_processing_helper = UserProcessingHelper.from_settings()

    def __init__(self, denied_operator_view_roles, denied_data_view_roles):
        self._denied_operator_view_roles = denied_operator_view_roles
        self._denied_data_view_roles = denied_data_view_roles

    @classmethod
    def from_settings(cls):
        denied_operator_view_roles = cars.settings.REQUEST_AGGREGATOR['callcenter']['denied_operator_view_roles']
        denied_data_view_roles = cars.settings.REQUEST_AGGREGATOR['callcenter']['denied_data_view_roles']
        return cls(denied_operator_view_roles, denied_data_view_roles)

    def check_request_type(self, request_type_collection=None):
        return not request_type_collection or self.REQUEST_TYPE.value in request_type_collection

    def get_dialog_url(self, dialog_id=None):
        if dialog_id is not None:
            return self.DATA_URL_TEMPLATE.format(dialog_id=dialog_id)
        return None

    def get_requests(self, request=None, limit=None, **kwargs):
        if self._check_phone_not_matched(**kwargs):
            return []

        filters = self._make_filters(**kwargs)

        if not filters and limit is None:
            raise Exception('at least one filter must be provided')

        target_message_limit = self.ESTIMATED_DIALOG_SIZE * limit if limit is not None else None

        target_messages = (
            Chat2DeskMessageEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(**filters)
            .order_by('-time_id')
            [:target_message_limit]
            .values('related_client', 'time_id')
        )

        grouped_target_message_times = collection_to_mapping(
            target_messages,
            item_key='related_client',
            value=lambda item: item['time_id']
        )

        if not grouped_target_message_times:
            return []

        related_clients = grouped_target_message_times.keys()

        min_request_time = min(g[-1] for g in grouped_target_message_times.values())
        min_request_time -= datetime.timedelta(seconds=self.MAX_REQUEST_END_GAP)

        max_request_time = max(g[0] for g in grouped_target_message_times.values())
        max_request_time += datetime.timedelta(seconds=self.MAX_REQUEST_END_GAP)

        dialog_bound_limit = 2 * limit if limit is not None else None  # start and end entries

        target_requests = (
            Chat2DeskMessageEntry.objects
            .using(cars.settings.DB_RO_ID)
            .prefetch_related('related_client', 'related_operator')
            .filter(
                entry_type__in=(Chat2DeskEntryType.REQUEST_START.value, Chat2DeskEntryType.REQUEST_END.value),
                time_id__gte=min_request_time,
                time_id__lte=max_request_time,
                related_client__in=related_clients,
            )
            .order_by('-time_id', '-id')
            .only('related_client', 'time_id', 'id', 'related_operator', 'transport')
            [:dialog_bound_limit]
        )

        grouped_requests = collection_to_mapping(target_requests, key=lambda item: item.related_client.id)

        flattened_requests = sorted(
            itertools.chain.from_iterable(
                self._group_requests(
                    reversed(grouped_requests[k]),
                    reversed(grouped_target_message_times.get(k, []))
                )
                for k in grouped_requests
            ),
            key=lambda x: x[0].time_id,
            reverse=True,
        )

        chat_dialogs = (
            Chat2DeskDialogEntry.objects.using(cars.settings.DB_RO_ID)
            .filter(related_client__in=related_clients)
        )

        grouped_chat_dialogs = collection_to_mapping(chat_dialogs, attr_key=('related_client', 'transport'))

        users_mapping = self._make_request_users_mapping(related_clients, request, kwargs)

        access_mapping = self._make_access_mapping(request)

        entries = self._transform_requests(flattened_requests, grouped_chat_dialogs, users_mapping, access_mapping, limit)
        return entries

    def _check_phone_not_matched(self, user=None, phone_number=None, **kwargs):
        # user phone number is not specified neither explicitly nor present in user info
        return user is not None and phone_number is None

    def _make_filters(self, since=None, until=None, phone_number=None, staff_entry_binding=None, **kwargs):
        filters = {}

        if phone_number is not None:
            clients = (
                Chat2DeskClientEntry.objects.using(cars.settings.DB_RO_ID).filter(assigned_phone=phone_number)
            )
            filters['related_client__in'] = clients

        if since is not None:
            filters['time_id__gte'] = since

        if until is not None:
            filters['time_id__lt'] = until

        if staff_entry_binding is not None:
            filters['related_operator__staff_entry_binding'] = staff_entry_binding

        return filters

    def _make_request_users_mapping(self, related_client_ids, request, kwargs):
        if 'user' in kwargs:
            related_user = kwargs['user']
            related_user_phone = kwargs.get('phone_number', str(related_user.phone))

            raw_entry_phone_bindings = {None: related_user}
            self.user_processing_helper.filter_deleted_users(raw_entry_phone_bindings, request)

            entry_phone_bindings = self.user_processing_helper.format_users(
                default=raw_entry_phone_bindings[None], update_from_key='phone'
            )
            users_mapping = {
                client_id: entry_phone_bindings[related_user_phone]
                for client_id in related_client_ids
            }

        else:
            related_clients = (
                Chat2DeskClientEntry.objects.using(cars.settings.DB_RO_ID).filter(id__in=related_client_ids)
            )

            raw_entry_phone_bindings = self.user_processing_helper.get_entries_phone_bindings(
                related_clients, phone_number_field='assigned_phone'
            )
            self.user_processing_helper.filter_deleted_users(raw_entry_phone_bindings, request)

            entry_phone_bindings = self.user_processing_helper.format_users(raw_entry_phone_bindings, update_from_key='phone')
            users_mapping = {
                client.id: entry_phone_bindings[client.assigned_phone]
                for client in related_clients
            }

        return users_mapping

    def _make_access_mapping(self, request=None):
        access_mapping = {}

        if request is not None:
            has_access_to_operators = not self.saas_client.check_user_role(
                *self._denied_operator_view_roles, request=request, require_all=False
            )
            has_access_to_data = not self.saas_client.check_user_role(
                *self._denied_data_view_roles, request=request, require_all=False
            )
        else:
            has_access_to_operators = False
            has_access_to_data = False

        access_mapping['has_access_to_operators'] = has_access_to_operators
        access_mapping['has_access_to_data'] = has_access_to_data

        return access_mapping

    def _group_requests(self, chat_entries, target_message_times):
        start = end = None

        target_time_iter = iter(target_message_times)
        target_time = next(target_time_iter, None)

        for e in chat_entries:
            if e.entry_type == Chat2DeskEntryType.REQUEST_START.value:
                do_match, target_time = self._do_interval_match(start, end, target_time, target_time_iter)
                if do_match:
                    yield (start, end)
                start, end = e, None

            elif e.entry_type == Chat2DeskEntryType.REQUEST_END.value:
                if start is None:
                    # error_message = 'no start entry provided for a dialog (message id {})'.format(e.id)
                    # LOGGER.error(error_message)
                    pass

                end = e

        do_match, target_time = self._do_interval_match(start, end, target_time, target_time_iter)
        if do_match:
            yield (start, end)

    def _do_interval_match(self, start, end, target_time, target_time_iter):
        do_match = False

        if start is not None:
            while target_time is not None and target_time < start.time_id:
                target_time = next(target_time_iter, None)

            if target_time is not None and (end is None or target_time <= end.time_id):
                do_match = True

        return do_match, target_time

    def _transform_requests(self, grouped_requests, grouped_chat_dialogs, users_mapping, access_mapping, limit):
        response_entries = (
            self._make_request_entry(start, end, grouped_chat_dialogs, users_mapping, access_mapping)
            for start, end in grouped_requests
        )
        response_entries = itertools.islice(response_entries, limit)
        response_entries_as_mappings = [e._asdict() for e in response_entries]
        return response_entries_as_mappings

    def _make_request_entry(self, request_start, request_end, grouped_chat_dialogs, users_mapping, access_mapping):
        time_enter = request_start.time_id.timestamp()

        if request_end is not None:
            time_exit = request_end.time_id.timestamp()
            duration = time_exit - time_enter
        else:
            time_exit = duration = None

        duration_print = datetime_helper.duration_to_str(duration)

        time_connect = None  # time to connect is None or not provided at all

        _related_client = request_start.related_client

        user = users_mapping.get(_related_client.id, None) if _related_client else None

        phone = _related_client.assigned_phone
        phone = str(phone) if phone is not None else phone

        transport = request_start.transport

        dialog_id = request_start.related_dialog_id

        if dialog_id is None:
            if grouped_chat_dialogs is None:
                related_dialogs = (
                    Chat2DeskDialogEntry.objects.using(cars.settings.DB_RO_ID)
                    .filter(related_client=_related_client, transport=transport)
                )
                related_dialog = related_dialogs.first()
            else:
                related_dialogs = grouped_chat_dialogs.get((_related_client, transport), [])
                related_dialog = related_dialogs[0] if related_dialogs else None

            if related_dialog is not None:
                dialog_id = related_dialog.related_id

        if access_mapping['has_access_to_data']:
            data_url = self.get_dialog_url(dialog_id)
        else:
            data_url = None

        data = {'chat_url': data_url}

        related_operator = request_start.related_operator

        if related_operator is None and request_end is not None:
            related_operator = request_end.related_operator

        agent_trait = related_operator.email if related_operator is not None else None
        agent_info_entry = self.staff_info_helper.get_agent_entry(email=agent_trait)
        agent = self.staff_info_helper.format_staff_entry(agent_info_entry)

        if access_mapping['has_access_to_operators']:
            operators = [agent]
        else:
            operators = []

        # consider provide internal id
        request_description = {
            'id': dialog_id,
            'origin': RequestOriginType.CHAT2DESK_EXT_DIALOG_ID.name,
            'queue': transport,
            'type': self.REQUEST_TYPE.value,
        }

        entry = AggregatedResponseEntry(
            time_enter=time_enter, time_exit=time_exit, time_connect=time_connect,
            duration=duration, duration_print=duration_print,
            phone=phone, operators=operators,
            tags=[], user=user, request=request_description,
            data_url=data_url, data=data, message=None,
            status=None, connect_trial_count=None
        )
        return entry
