import logging

from itertools import groupby

from django.db import transaction
from django.db.models import Q

from cars.core.util import phone_number_helper
from cars.callcenter.core import StaffInfoHelper
from cars.request_aggregator.models.chat2desk_stats import (
    Chat2DeskClientEntry, Chat2DeskMessageEntry, Chat2DeskOperatorEntry, Chat2DeskEntryType
)

from .collecting_helper import Chat2DeskCollectingHelper

LOGGER = logging.getLogger(__name__)


class Chat2DeskServiceHelper(Chat2DeskCollectingHelper):
    MAX_CLIENTS_DB_BATCH_SIZE = 2000  # limit of clients to request from DB
    MAX_MESSAGES_DB_BATCH_SIZE = 2000  # limit of messages to request from DB

    staff_info_helper = StaffInfoHelper.make_default()

    def process_absent_entries(self, start_offset=0, count=None, process_clients=True, process_dialogs=True):
        total_entries_processed = 0

        limit = self._api_helper.MAX_CLIENTS_BATCH_SIZE

        if count is None:
            end_offset = self._api_helper.get_total_clients_count()

            if end_offset is None:
                LOGGER.error('cannot obtain total clients count using chat2desk API')
                end_offset = start_offset  # empty range
        else:
            end_offset = start_offset + count

        for offset in range(start_offset, end_offset, limit):
            request_limit = min(limit, end_offset - offset)
            clients = self._api_helper.get_clients(offset, request_limit)

            if clients is None:
                LOGGER.error('cannot obtain clients using chat2desk API')
                break

            client_dialog_mapping = {c['id']: (None, None) for c in clients}
            shared_dialog_info_mapping = {}

            processed_client_entries = processed_dialog_entries = 0

            with transaction.atomic(savepoint=False):
                if process_clients:
                    processed_client_entries = self._process_clients(client_dialog_mapping, shared_dialog_info_mapping)

                if process_dialogs:
                    processed_dialog_entries = self._process_dialogs(client_dialog_mapping, shared_dialog_info_mapping)

            total_entries_processed += len(processed_client_entries)
            total_entries_processed += len(processed_dialog_entries)

        return total_entries_processed

    def process_absent_clients(self, start_offset=0, count=None):
        return self.process_absent_entries(start_offset=start_offset, count=count,
                                           process_clients=True, process_dialogs=False)

    def process_absent_dialogs(self, start_offset=0, count=None):
        return self.process_absent_entries(start_offset=start_offset, count=count,
                                           process_clients=False, process_dialogs=True)

    def update_mocked_clients(self):
        total_entries = updated_entries = 0

        with transaction.atomic(savepoint=False):
            mocked_clients = Chat2DeskClientEntry.objects.select_for_update().filter(
                Q(extra_identification_info__contains=self._api_helper.MOCKED_CLIENT_RU_INFO)
                | Q(assigned_phone='+NoneNone')
            )

            for mocked_client in mocked_clients:
                client_id = mocked_client.related_id
                client_updates = list(self._process_client_entry(client_id, None))

                if client_updates:
                    first_client = client_updates[0]

                    mocked_client.assigned_phone = first_client.assigned_phone
                    mocked_client.extra_identification_info = first_client.extra_identification_info
                    mocked_client.save()

                    updated_entries += 1

                Chat2DeskClientEntry.objects.bulk_create(client_updates[1:])
                total_entries += 1

        LOGGER.info('info about {} mocked clients of {} total has been successfully updated'
                    .format(updated_entries, total_entries))

        return total_entries, updated_entries

    def update_client_info_from_messages(self, start_offset=0, count=None):
        total_entries_processed = 0

        limit = self.MAX_MESSAGES_DB_BATCH_SIZE

        if count is None:
            end_offset = Chat2DeskMessageEntry.objects.count() - start_offset
        else:
            end_offset = start_offset + count

        for offset in range(start_offset, end_offset, limit):
            request_limit = min(limit, end_offset - offset)
            message_entries = (
                Chat2DeskMessageEntry.objects.select_related('related_client')
                .order_by('id')
                [offset:offset + request_limit]
            )
            message_related_client_id_mapping = {
                m.related_client.related_id: m.related_client.id
                for m in message_entries
            }

            client_entries = list(
                Chat2DeskClientEntry.objects
                .filter(related_id__in=message_related_client_id_mapping)
                .order_by('related_id', 'id')
            )

            grouped_related_clients = {
                message_related_client_id_mapping[cid]: list(g) for cid, g in
                groupby(client_entries, key=lambda x: x.related_id)
            }

            for message_entry in message_entries:
                raw_message_entry = {'type': message_entry.entry_type, 'text': message_entry.text}

                message_time_id = message_entry.time_id

                cid = message_entry.related_client_id

                all_related_clients = grouped_related_clients[cid]
                client_entry = all_related_clients[0]

                self._try_update_client_info(raw_message_entry, message_time_id, client_entry, all_related_clients)

            total_entries_processed += request_limit

        return total_entries_processed

    def rebind_messages_to_registration_client_entry(self, start_offset=0, count=None):
        total_entries_processed = 0

        limit = self.MAX_CLIENTS_DB_BATCH_SIZE

        if count is None:
            end_offset = Chat2DeskClientEntry.objects.count() - start_offset
        else:
            end_offset = start_offset + count

        for offset in range(start_offset, end_offset, limit):
            request_limit = min(limit, end_offset - offset)
            client_entries = (
                Chat2DeskClientEntry.objects
                .order_by('related_id', 'id')
                [offset:offset + request_limit]
            )
            grouped_client_entries = groupby(client_entries, lambda x: x.related_id)

            for _, group in grouped_client_entries:
                registration_entry = next(group)
                extra_entries = list(group)

                if extra_entries:
                    with transaction.atomic(savepoint=False):
                        messages_to_rebind = (
                            Chat2DeskMessageEntry.objects.select_for_update()
                            .filter(related_client__in=extra_entries)
                        )

                        for message in messages_to_rebind:
                            message.related_client = registration_entry
                            message.save()

            total_entries_processed += request_limit

        return total_entries_processed

    def update_dialogs_bounds(self, start_offset=0, count=None):
        total_entries_processed = 0

        limit = self.MAX_CLIENTS_DB_BATCH_SIZE

        if count is None:
            end_offset = Chat2DeskClientEntry.objects.count() - start_offset
        else:
            end_offset = start_offset + count

        for offset in range(start_offset, end_offset, limit):
            request_limit = min(limit, end_offset - offset)
            client_entries = (
                Chat2DeskClientEntry.objects
                .order_by('id')
                [offset:offset + request_limit]
            )

            for client_entry in client_entries:
                client_message_count = Chat2DeskMessageEntry.objects.filter(related_client=client_entry).count()

                if not client_message_count:
                    continue

                last_processed_message = None

                message_limit = self.MAX_MESSAGES_DB_BATCH_SIZE

                for message_offset in range(0, client_message_count, message_limit):
                    message_request_limit = min(message_limit, client_message_count - message_offset)

                    with transaction.atomic(savepoint=False):
                        client_messages = (
                            Chat2DeskMessageEntry.objects.select_for_update()
                            .filter(related_client=client_entry)
                            .order_by('id')
                            [message_offset:message_offset + message_request_limit]
                        )

                        for message in client_messages:
                            message_values = {
                                'type': message.entry_type,
                                'text': message.text,
                                'id': message.related_id,
                                'client_id': client_entry.related_id,
                            }
                            message_history = [last_processed_message] if last_processed_message else []

                            if self._api_helper.check_message_request_end(message_values):
                                entry_type = Chat2DeskEntryType.REQUEST_END.value
                            elif (message.entry_type == Chat2DeskEntryType.FROM_CLIENT.value and
                                  self._api_helper.check_prev_message_request_end(message_values, message_history)):
                                entry_type = Chat2DeskEntryType.REQUEST_START.value
                            else:
                                entry_type = None

                            if entry_type is not None and entry_type != message.entry_type:
                                message.entry_type = entry_type
                                message.save()

                            last_processed_message = message

                total_entries_processed += client_message_count

        return total_entries_processed

    def update_bad_formatted_phone_numbers(self, start_offset=0, count=None):
        total_entries_processed = 0

        limit = self.MAX_CLIENTS_DB_BATCH_SIZE

        if count is None:
            end_offset = Chat2DeskClientEntry.objects.count() - start_offset
        else:
            end_offset = start_offset + count

        for offset in range(start_offset, end_offset, limit):
            request_limit = min(limit, end_offset - offset)

            with transaction.atomic(savepoint=False):
                client_entries = (
                    Chat2DeskClientEntry.objects
                    .select_for_update()
                    .order_by('id')
                    [offset:offset + request_limit]
                )

                for entry in client_entries:
                    assigned_phone = entry.assigned_phone

                    if not assigned_phone:
                        continue

                    origin_phone = str(assigned_phone).replace('+', '98')

                    normalized_phone = phone_number_helper.normalize_phone_number(origin_phone)

                    if normalized_phone is None or assigned_phone != normalized_phone:
                        LOGGER.info('update number "{}" to "{}"'.format(assigned_phone, normalized_phone))
                        entry.assigned_phone = normalized_phone
                        entry.save()

            total_entries_processed += request_limit

        return total_entries_processed

    def update_staff_bindings(self, **kwargs):
        with transaction.atomic(savepoint=False):
            entries = (
                Chat2DeskOperatorEntry.objects.select_for_update()
                .filter(
                    staff_entry_binding__isnull=True,
                    email__isnull=False,
                )
            )

            for entry in entries:
                agent_entry = self.staff_info_helper.get_agent_entry(email=entry.email)
                if agent_entry is not None:
                    entry.staff_entry_binding = agent_entry
                    entry.save()

        total_numbers = len(entries)
        LOGGER.info('total numbers updated: {}'.format(total_numbers))

        return total_numbers
