import copy
import hashlib
import logging

from django.db import transaction
from django.utils import timezone

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.models.chat2desk_stats import (
    Chat2DeskOperatorEntry, Chat2DeskClientEntry, Chat2DeskDialogEntry,
    Chat2DeskAttachmentType, Chat2DeskEntryType
)

from .collecting_helper import Chat2DeskCollectingHelperBase

LOGGER = logging.getLogger(__name__)


class Chat2DeskWebHookHelper(Chat2DeskCollectingHelperBase):
    def __init__(self, robot_login, robot_auth_token, api_token, request_timeout=15, retries=3):
        super().__init__(api_token, request_timeout, retries)
        self._robot_login = robot_login
        self._robot_auth_token = robot_auth_token

        self._hook_mapping = {
            'new_client': self._process_new_client,
            'inbox': self._process_new_ordinary_message,
            'outbox': self._process_new_ordinary_message,
            'close_dialog': self._process_dialog_close,
        }

        self._content_type_mapping = {
            'video': Chat2DeskAttachmentType.VIDEO.value,
            'audio': Chat2DeskAttachmentType.AUDIO.value,
            'image': Chat2DeskAttachmentType.PHOTO.value,
        }

        self._default_content_type = Chat2DeskAttachmentType.PDF.value

    @classmethod
    def from_settings(cls):
        chat2desk_settings = settings['chat2desk']
        robot_login = chat2desk_settings['robot_username']
        robot_auth_token = chat2desk_settings['robot_auth_token']
        api_token = chat2desk_settings['api_token']
        request_timeout = chat2desk_settings['request_timeout']
        return cls(robot_login, robot_auth_token, api_token, request_timeout)

    def authorize_request(self, request):
        token_hash = request.query_params.get('token_hash', None)

        if token_hash is None:
            return False

        expected_token_hash = hashlib.sha256(self._robot_auth_token.encode('ascii')).hexdigest()

        is_authorized = (expected_token_hash == token_hash)
        return is_authorized

    def _request_client_update(self, message_entry):
        raw_client = message_entry.get('client', None)
        client_id = message_entry['client_id']
        time_id = timezone.now()
        client_dialog_info = self._api_helper.DialogMetaInfo(client_id, time_id, None, None, None)

        client_entries = self._process_client_entry(client_id, client_dialog_info, raw_client=raw_client)

        if client_entries is not None:
            client_entries = list(client_entries)
            Chat2DeskClientEntry.objects.bulk_create(client_entries)
            LOGGER.info('chat2desk clients: {} extra entries created for client {}, assigned phones - {}'
                        .format(len(client_entries), client_id, [str(e.assigned_phone) for e in client_entries]))

        updated_clients = list(Chat2DeskClientEntry.objects.filter(related_id=client_id).order_by('id'))
        return updated_clients

    def _request_dialog_update(self, message_entry, client_entries):
        if client_entries:
            client_entry = client_entries[0]
            dialog_id = message_entry['dialog_id']
            transport = message_entry['transport']

            with transaction.atomic(savepoint=False):
                updated_dialog = Chat2DeskDialogEntry.objects.filter(
                    related_client=client_entry,
                    related_id=dialog_id,
                    transport=transport,
                ).first()

                # to be done: add unique constraint and integrity exception check
                if updated_dialog is None:
                    updated_dialog = Chat2DeskDialogEntry.objects.create(
                        related_client=client_entry,
                        related_id=dialog_id,
                        transport=transport,
                    )

        else:
            updated_dialog = None

        return updated_dialog

    def _request_operator_update(self, message_entry):
        total_saved_operators = Chat2DeskOperatorEntry.objects.all().count()
        raw_operator_entries, _ = self._api_helper.get_remaining_operators(total_saved_operators)

        if raw_operator_entries is not None:
            processed_entries = [self._process_operator_entry(e) for e in raw_operator_entries]
            Chat2DeskOperatorEntry.objects.bulk_create(processed_entries)

        operator_id = message_entry['operator_id']
        updated_operator = Chat2DeskOperatorEntry.objects.filter(related_id=operator_id).first()
        return updated_operator

    def process_entry(self, entry):
        hook_type = entry['hook_type']
        hook_handler = self._hook_mapping.get(hook_type, None)
        if hook_handler is not None:
            hook_handler(entry)
        else:
            LOGGER.warning('no handler found for an entry: {}'.format(entry))
        return hook_type

    def _process_new_client(self, raw_entry):
        time_id = timezone.now()
        client_id = raw_entry['id']

        client_dialog_info = self._api_helper.DialogMetaInfo(client_id, time_id, None, None, None)

        client_entries = self._process_client_entry(client_id, client_dialog_info, raw_client=raw_entry)

        Chat2DeskClientEntry.objects.bulk_create(client_entries)

    def _process_new_ordinary_message(self, raw_entry):
        raw_entry['time_id'] = timezone.now()
        raw_entry['type'] = self._get_message_entry_type(raw_entry)

        attachments = raw_entry['attachments']

        if len(attachments) > 1:  # make multiple message entries
            for attachment in attachments:
                raw_entry_copy = copy.deepcopy(raw_entry)
                raw_entry_copy['attachments'] = [attachment]
                self._process_new_ordinary_message(raw_entry_copy)
        else:
            if attachments:
                self._set_attachment(raw_entry)

            message_entry = self._process_message_entry(raw_entry)
            if message_entry is not None:
                message_entry.save()

    def _get_message_entry_type(self, message_entry):
        if message_entry['type'] is None:
            entry_type = Chat2DeskEntryType.SYSTEM.value  # e.g. internal comments
        elif self._api_helper.check_message_request_end(message_entry):
            entry_type = Chat2DeskEntryType.REQUEST_END.value
        elif (message_entry['type'] == Chat2DeskEntryType.FROM_CLIENT.value and
              message_entry['is_new_request']):
            entry_type = Chat2DeskEntryType.REQUEST_START.value
        else:
            entry_type = message_entry['type']

        return entry_type

    def _set_attachment(self, raw_entry):
        # at least fix webhook 'video' key processing
        attachment = raw_entry['attachments'][0]
        attachment_type = self._detect_attachment_type(attachment['content_type'])
        if raw_entry.get(attachment_type, None) is None:
            raw_entry[attachment_type] = attachment

    def _detect_attachment_type(self, content_type):
        content_type_prefix = content_type.split('/')[0]
        content_key = self._content_type_mapping.get(content_type_prefix, self._default_content_type)
        return content_key

    def _process_dialog_close(self, raw_entry):
        raw_entry['time_id'] = timezone.now()
        raw_entry['type'] = Chat2DeskEntryType.REQUEST_END.value

        dialog_id = raw_entry['dialog_id']
        related_dialog = None

        if 'client_id' not in raw_entry:
            related_dialog = (
                Chat2DeskDialogEntry.objects.select_related('related_client')
                .filter(related_id=dialog_id)
                .first()
            )

            if related_dialog is not None:
                raw_entry['client_id'] = related_dialog.related_client.related_id
            else:
                raise Exception('no dialog with specified id can be found: id - {}'.format(dialog_id))

        raw_entry.setdefault('message_id', 0)  # mock value
        raw_entry.setdefault('text', 'Диалог был закрыт (webhook)')

        message_entry = self._process_message_entry(raw_entry, related_dialog=related_dialog)
        if message_entry is not None:
            message_entry.save()
