import logging

from django.db.models import Max

import cars.settings

from cars.callcenter.core.staff_helper import StaffInfoHelper

from cars.request_aggregator.core.common_helper import YTHelper
from cars.request_aggregator.core.request_time_sync_helper import RequestStateSyncHelper

LOGGER = logging.getLogger(__name__)


class DataExportingSyncHelper(RequestStateSyncHelper):
    entry_model_cls = None

    MAX_ENTRIES_COUNT = 5000

    @classmethod
    def from_settings(cls, sync_status_origin):
        return cls(
            stat_sync_origin=sync_status_origin,
            default_state={'next_entry_id': 1}
        )

    def _get_state(self, sync_entry):
        entry_id_lo = sync_entry.data['next_entry_id']
        max_entry_id = self._get_qs().aggregate(Max('id')).get('id__max', 0)
        entry_id_hi = min(entry_id_lo + self.MAX_ENTRIES_COUNT, max_entry_id + 1)
        return entry_id_lo, entry_id_hi

    def _get_qs(self):
        return self.entry_model_cls.objects.using(cars.settings.DB_RO_ID).all()

    def _update_state(self, sync_entry, state=None):
        entry_id_lo, entry_id_hi = state

        sync_entry.data['next_entry_id'] = entry_id_hi

        # try to get more entries
        has_more = (entry_id_hi - entry_id_lo) == self.MAX_ENTRIES_COUNT
        return has_more


class DataExportingHelper(object):
    _sync_helper_cls = None
    _entry_model_cls = None
    MAX_ENTRIES_COUNT = None

    def __init__(self, sync_origin, export_table_path):
        self._staff_helper = StaffInfoHelper.make_default()

        self._sync_helper = self._sync_helper_cls.from_settings(sync_origin)
        self._yt_helper = YTHelper(export_table_path)

    @classmethod
    def make(cls, sync_origin, export_table_path):
        cls._entry_model_cls = cls._sync_helper_cls.entry_model_cls
        cls.MAX_ENTRIES_COUNT = cls._sync_helper_cls.MAX_ENTRIES_COUNT
        return cls(sync_origin, export_table_path)

    def perform(self):
        for sync_state in self._sync_helper.iter_states_to_process():
            data_to_store = self._iter_data_to_store(sync_state)
            self._yt_helper.export_data(data_to_store)

    def optimize(self):
        self._yt_helper.merge_chunks()

    def _iter_data_to_store(self, sync_state):
        entry_id_lo, entry_id_hi = sync_state

        if entry_id_lo < entry_id_hi:
            entries = self._iter_entries(entry_id_lo, entry_id_hi)
            yield from (self._make_export_entry(entry) for entry in entries)

    def _iter_entries(self, entry_id_lo, entry_id_hi):
        entries = self._entry_model_cls.objects.using(cars.settings.DB_RO_ID).filter(
            id__gte=entry_id_lo,
            id__lt=entry_id_hi,
        )
        yield from entries

    def _make_export_entry(self, entry):
        raise NotImplementedError

    def time_to_repr(self, value):
        return int(value.timestamp() * 1000) if value is not None else None

    def get_original_phone(self, entry, *, meta_info_key='original_phone'):
        if entry.phone:
            original_phone = str(entry.phone)
        elif entry.meta_info:
            original_phone = entry.meta_info.get(meta_info_key, None)
        else:
            original_phone = None
        return original_phone

    def get_agent_username(self, entry):
        agent_username = None
        if entry.staff_entry_binding_id is not None:
            agent_entry = self._staff_helper.get_agent_entry(id=entry.staff_entry_binding_id)
            if agent_entry is not None:
                agent_username = agent_entry.username
        return agent_username
