import datetime
import logging

from django.db import transaction

from cars.core.util import datetime_helper, phone_number_helper
from cars.callcenter.models.call_assignment import CallAssignment, CallAssignmentCallCenter
from cars.users.models.user import User

from cars.request_aggregator.models.internal_cc_stats import InternalCCVerb

from .api_helper import TelephonyAssignmentApiHelper
from .syncing_helper import TelephonyAssignmentRequestTimeSyncHelper

from ..base.collecting_helper import CollectingHelperBase
from cars.request_aggregator.core.internal_cc.mixins import InternalCCReporterMixin

LOGGER = logging.getLogger(__name__)


class TelephonyAssignmentCollectingHelper(CollectingHelperBase, InternalCCReporterMixin):
    HANG_CALL_DURATION = datetime.timedelta(minutes=45)  # forcibly terminate calls interpreted as hung

    @classmethod
    def from_settings(cls):
        return cls(
            api_helper=TelephonyAssignmentApiHelper.from_settings(),
            time_sync_helper=TelephonyAssignmentRequestTimeSyncHelper.from_settings()
        )

    def _filter_already_stored_data(self, raw_entries, start_timestamp, end_timestamp):
        return raw_entries

    def _process_entries(self, data, entries_stat):
        entries_to_register = {}
        entries_to_connect = {}
        entry_ids_to_remove = []

        for raw_entry in data:
            assert isinstance(raw_entry, self._api_helper.API_RESPONSE)

            verb_instance = InternalCCVerb.try_make(raw_entry.verb)

            if verb_instance.is_enter_entry():
                self._add_register_call_entry(entries_to_register, raw_entry)
                entries_stat['enter_queue'] += 1

            elif verb_instance.is_connect_entry():
                self._add_connect_call_entry(entries_to_connect, raw_entry)
                entries_stat['connect'] += 1

            elif (
                    verb_instance.is_completion_entry() or
                    verb_instance.is_abandon_entry() or
                    verb_instance.is_transfer_entry()
            ):
                self._add_remove_call_entry(entry_ids_to_remove, raw_entry)
                entries_stat['exit_queue'] += 1

            else:
                # e.g. RING_NO_ANSWER is not processed
                pass

        return entries_to_register, entries_to_connect, entry_ids_to_remove

    _process_entry = None

    def _add_register_call_entry(self, entries_to_register, raw_entry):
        call_id = raw_entry.call_id
        time_id = raw_entry.time_id
        from_number = phone_number_helper.normalize_phone_number(raw_entry.data)
        queue = raw_entry.queue

        if from_number:
            # do not add bad formatted entries
            from_user = (
                User.objects.filter(phone=from_number)
                .order_by('status')  # make "active" first if exists
                .first()
            )
            entries_to_register[call_id] = (time_id, from_number, from_user, queue)

    def _add_connect_call_entry(self, entries_to_connect, raw_entry):
        call_id = raw_entry.call_id
        connected_at = raw_entry.time_id
        to_number = int(raw_entry.agent.split('/')[1])
        _to_user_binding = self.staff_info_helper.get_agent_entry(work_phone=to_number)
        to_user = _to_user_binding.user if _to_user_binding is not None else None
        entries_to_connect[call_id] = (connected_at, to_number, to_user)

    def _add_remove_call_entry(self, entries_to_remove, raw_entry):
        call_id = raw_entry.call_id
        entries_to_remove.append(call_id)

    def _apply_entries(self, entries):
        raw_entries_to_register, raw_entries_to_connect, entry_ids_to_remove = entries

        for call_id in entry_ids_to_remove:
            raw_entries_to_register.pop(call_id, None)
            raw_entries_to_connect.pop(call_id, None)

        with transaction.atomic(savepoint=False):
            if entry_ids_to_remove:
                entries_to_remove = CallAssignment.objects.filter(call_id__in=entry_ids_to_remove)
                self._remove_call_assignment_entries_in_tags(entries_to_remove, extra_mark='collector')
                entries_to_remove.delete()

            if raw_entries_to_register:
                existing_call_ids = CallAssignment.objects.values_list('call_id', flat=True)
                call_center = CallAssignmentCallCenter.Yandex.value
                assignment_objects = [
                    CallAssignment(
                        call_id=call_id,
                        added_at=time_id,
                        from_number=from_number,
                        from_user=from_user,
                        call_center=call_center,
                        queue=queue
                    )
                    for (call_id, (time_id, from_number, from_user, queue)) in raw_entries_to_register.items()
                    if call_id is None or call_id not in existing_call_ids
                ]
                CallAssignment.objects.bulk_create(assignment_objects)
                self._upsert_call_assignment_entries_in_tags(assignment_objects, extra_mark='collector')

            if raw_entries_to_connect:
                entries_to_update = (CallAssignment.objects.select_for_update()
                                     .filter(call_id__in=raw_entries_to_connect, connected_at__isnull=True))

                for e in entries_to_update:
                    related_entry = raw_entries_to_connect[e.call_id]
                    e.connected_at, e.to_number, e.to_user = related_entry
                    e.save()

                self._upsert_call_assignment_entries_in_tags(entries_to_update, extra_mark='collector')

    def process_hung_entries(self):
        hung_call_start_time_upper_bound = datetime_helper.utc_now() - self.HANG_CALL_DURATION
        hung_entries = CallAssignment.objects.filter(added_at__lte=hung_call_start_time_upper_bound)

        hung_entries_count = len(hung_entries)

        if hung_entries_count:
            entry_phone_numbers = [str(e.from_number) for e in hung_entries]

            self._remove_call_assignment_entries_in_tags(hung_entries, extra_mark='hung_processor')
            hung_entries.delete()

            LOGGER.warning(
                'removed {} hung entries, phone numbers - {}'.format(hung_entries_count, entry_phone_numbers)
            )

        return hung_entries_count
