import logging

from cars.core.telephony import TelephonyCallCenter
from cars.callcenter.core import StaffInfoHelper
from cars.callcenter.models.call_assignment import CallAssignment, CallAssignmentCallCenter
from cars.users.models import User

from cars.request_aggregator.models.call_center_common import CallRoutingEntry, RoutingVerb
from cars.request_aggregator.core.internal_cc.mixins import InternalCCReporterMixin

from ..base.webhook_helper import RegisteringHelperBase


LOGGER = logging.getLogger(__name__)


class AssignmentRegisteringHelper(RegisteringHelperBase, InternalCCReporterMixin):
    staff_info_helper = StaffInfoHelper.make_default()

    def _process_entries(self, raw_entries, entries_stat):
        related_call_routing_entries = {
            (x.call_id, x.verb): x for x in
            CallRoutingEntry.objects.filter(call_id__in=(a.call_id for a in raw_entries))
        }
        entries = (
            self._process_entry(connect_entry, related_call_routing_entries) for connect_entry in raw_entries
            if self._handle_entry(connect_entry, entries_stat, related_call_routing_entries)
        )
        return entries

    def _handle_entry(self, connect_entry, entries_stat, related_call_routing_entries):
        # verb is XCONNECT, data is call waiting time (time from time enter),
        #   queue is one of internal ones (carsharing, carsharing-spb or carsharing-vip),
        #   call center is yandex for all calls now
        # base verb stat is processed in other helper
        call_id = connect_entry.call_id

        incoming_entry = related_call_routing_entries.pop((call_id, RoutingVerb.INCOMING.value), None)
        distribute_entry = related_call_routing_entries.pop((call_id, RoutingVerb.DISTRIBUTE.value), None)

        if incoming_entry is None and distribute_entry is None:
            LOGGER.warning(
                'incoming and distribute entries for call {} are not found.'
                ' source user cannot be determined'.format(call_id)
            )
            self._increase_not_matched_stat(connect_entry, entries_stat)
            return False

        if distribute_entry is not None:
            enter_entry = distribute_entry
        else:
            enter_entry = incoming_entry
            LOGGER.warning(
                'distribute entry for call {} is not found, however incoming entry presents'.format(call_id)
            )
            self._increase_no_distribute_stat(connect_entry, entries_stat)

        related_call_routing_entries[call_id] = enter_entry

        if not enter_entry.phone:
            LOGGER.warning(
                'enter entry for call {} has invalid phone number. source user cannot be determined'.format(call_id)
            )
            self._increase_bad_phone_stat(connect_entry, entries_stat)
            return False

        if enter_entry.target_cc != TelephonyCallCenter.YANDEX.value:
            LOGGER.warning(
                'call assignment entries are expected from Yandex callcenter only (call id: {})'.format(call_id)
            )

        return True

    def _increase_no_distribute_stat(self, connect_entry, entries_stat):
        stat_name = '{}_NO_DISTRIBUTE'.format(connect_entry.verb)
        entries_stat[stat_name] += 1

    def _increase_not_matched_stat(self, connect_entry, entries_stat):
        stat_name = '{}_NOT_MATCHED'.format(connect_entry.verb)
        entries_stat[stat_name] += 1

    def _increase_bad_phone_stat(self, connect_entry, entries_stat):
        stat_name = '{}_BAD_PHONE'.format(connect_entry.verb)
        entries_stat[stat_name] += 1

    def _process_entry(self, connect_entry, related_call_routing_entries):
        call_id = connect_entry.call_id

        enter_entry = related_call_routing_entries[call_id]

        call_center = CallAssignmentCallCenter.Yandex.value
        queue = connect_entry.queue

        from_number = enter_entry.phone

        from_user = (
            User.objects.filter(phone=from_number)
            .order_by('status')  # make "active" first if exists
            .first()
        )

        to_number = self.staff_info_helper.try_extract_agent_work_phone(connect_entry.agent)
        _to_user_binding = self.staff_info_helper.get_agent_entry(work_phone=to_number)
        to_user = _to_user_binding.user if _to_user_binding is not None else None

        processed_entry = CallAssignment(
            from_number=from_number, to_number=to_number,
            from_user=from_user, to_user=to_user,
            added_at=enter_entry.time_id, connected_at=connect_entry.time_id + connect_entry.data,
            call_id=call_id, call_center=call_center, queue=queue
        )
        return processed_entry

    def _apply_entries(self, entries):
        entries = list(entries)
        CallAssignment.objects.bulk_create(entries)
        self._upsert_call_assignment_entries_in_tags(entries, extra_mark='webhook')
