import collections
import logging

from cars.core.telephony import TimeRangeTelephonyApiHelper
from cars.callcenter.core import StaffInfoHelper

from cars.request_aggregator.core.request_time_sync_helper import RequestTimeSyncHelper

LOGGER = logging.getLogger(__name__)


class CollectingHelperBase(object):
    staff_info_helper = StaffInfoHelper.make_default()

    def __init__(self, api_helper, time_sync_helper):
        assert isinstance(api_helper, TimeRangeTelephonyApiHelper)
        assert isinstance(time_sync_helper, RequestTimeSyncHelper)
        self._api_helper = api_helper
        self._time_sync_helper = time_sync_helper

    def update_data(self, batch_size=None):
        all_entries_stat = collections.Counter()

        for start_timestamp, end_timestamp in self._time_sync_helper.iter_time_span_to_process():
            LOGGER.info('processing time range from {} to {}'.format(start_timestamp, end_timestamp))

            entries_stat = collections.Counter()
            data = self._api_helper.get_time_range_data(start_timestamp, end_timestamp)

            if data is not None:
                raw_entries = self._filter_already_stored_data(data, start_timestamp, end_timestamp)
                entries = self._process_entries(raw_entries, entries_stat)

                batch = None
                try:
                    if batch_size is None:
                        batch = entries
                        self._apply_entries(batch)
                    else:
                        entries = list(entries)
                        for offset in range(0, len(entries), batch_size):
                            batch = entries[offset:offset + batch_size]
                            self._apply_entries(batch)
                except Exception:
                    self._handle_applying_error(batch)
                    raise

                LOGGER.info('processed entries: {}'.format(entries_stat))
                all_entries_stat.update(entries_stat)
            else:
                LOGGER.warning('error occurred requesting data')

        return all_entries_stat

    def _filter_already_stored_data(self, raw_entries, start_timestamp, end_timestamp):
        return raw_entries

    def _process_entries(self, raw_entries, entries_stat):
        entries = (
            self._process_entry(entry) for entry in raw_entries
            if self._handle_entry(entry, entries_stat)
        )
        return entries

    def _handle_entry(self, raw_entry, entries_stat):
        entries_stat[raw_entry.verb] += 1
        return True

    def _process_entry(self, raw_entry):
        raise NotImplementedError

    def _apply_entries(self, entries):
        raise NotImplementedError

    def _handle_applying_error(self, entries):
        entries = list(entries) if not isinstance(entries, list) else entries
        if len(entries) > 10:
            LOGGER.error('Error applying entries: one of {} items in batch is incorrect'.format(len(entries)))
        else:
            LOGGER.error('Error applying entries: {}'.format('; '.join(str(entry) for entry in entries)))
