from cars.settings import REQUEST_AGGREGATOR as settings

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.callcenter.core import StaffInfoHelper

from cars.request_aggregator.core.phone_binding_helper import UserProcessingHelper
from cars.request_aggregator.core.request_tags import TagDescriptionHelper
from cars.request_aggregator.serializers.common import RequestType


class TelephonyProcessingHelperBase(object):
    REQUEST_TYPE = RequestType.phone

    DATA_URL_TEMPLATE = 'https://tel.yandex-team.ru/queuemetricscallrec?callid={call_id}'

    saas_client = SaasDriveAdminClient.from_settings()
    staff_info_helper = StaffInfoHelper.make_default()
    tag_description_helper = TagDescriptionHelper()
    user_processing_helper = UserProcessingHelper.from_settings()

    def __init__(self, call_track_view_actions, denied_operator_view_roles, denied_data_view_roles):
        self._call_track_view_actions = call_track_view_actions
        self._denied_operator_view_roles = denied_operator_view_roles
        self._denied_data_view_roles = denied_data_view_roles

    @classmethod
    def from_settings(cls):
        call_track_view_actions = settings['callcenter']['call_track_view_actions']
        denied_operator_view_roles = settings['callcenter']['denied_operator_view_roles']
        denied_data_view_roles = settings['callcenter']['denied_data_view_roles']
        return cls(call_track_view_actions, denied_operator_view_roles, denied_data_view_roles)

    def check_request_type(self, request_type_collection=None):
        return not request_type_collection or self.REQUEST_TYPE.value in request_type_collection

    def get_call_url(self, call_id):
        if call_id is not None:
            return self.DATA_URL_TEMPLATE.format(call_id=call_id)
        return None

    def get_requests(self, limit=None, **kwargs):
        raise NotImplementedError

    def _make_call_filters(self, phone_number=None, time_range=None, staff_entry_binding=None, **kwargs):
        raise NotImplementedError

    def _make_call_users_mapping(self, call_entries, request, kwargs):
        if 'user' in kwargs:
            raw_entry_phone_bindings = {None: kwargs['user']}
            self.user_processing_helper.filter_deleted_users(raw_entry_phone_bindings, request)

            entry_phone_bindings = self.user_processing_helper.format_users(
                default=raw_entry_phone_bindings[None], update_from_key='phone'
            )

        else:
            raw_entry_phone_bindings = self.user_processing_helper.get_entries_phone_bindings(call_entries)
            self.user_processing_helper.filter_deleted_users(raw_entry_phone_bindings, request)

            entry_phone_bindings = self.user_processing_helper.format_users(
                raw_entry_phone_bindings, update_from_key='phone'
            )

        users_mapping = {
            call_entry.id: entry_phone_bindings[call_entry.phone]
            for call_entry in call_entries
            if call_entry.phone is not None  # process ENTER entries only
        }

        return users_mapping

    def _make_access_mapping(self, request=None):
        access_mapping = {}

        if request is not None:
            has_access_to_call_id = (
                self.saas_client.check_user_action(
                    *self._call_track_view_actions, request=request, require_all=False
                ) and
                not self.saas_client.check_user_role(
                    *self._denied_data_view_roles, request=request, require_all=False
                )
            )
            has_access_to_operators = not self.saas_client.check_user_role(
                *self._denied_operator_view_roles, request=request, require_all=False
            )
        else:
            has_access_to_call_id = False
            has_access_to_operators = False

        access_mapping['has_access_to_call_id'] = has_access_to_call_id
        access_mapping['has_access_to_operators'] = has_access_to_operators

        return access_mapping
