import itertools

from django.utils import timezone

import cars.settings

from cars.callcenter.core import StaffInfoHelper, StaffStatAccessHelper
from cars.callcenter.models import CallCenterStaffEntry

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.core.common_helper import YTHelper


class OperatorsInfoHelper(object):

    MAX_UIDS_PER_REQUEST = 250  # request may be too long

    staff_info_helper = StaffInfoHelper.make_default()

    def __init__(self, export_table, operator_department_urls):
        self._yt_helper = YTHelper(export_table)
        self._staff_stat_access_helper = StaffStatAccessHelper()
        self._operator_department_urls = operator_department_urls

    @classmethod
    def from_settings(cls):
        operators_export_settings = settings['callcenter']['export']['operators']
        export_table = operators_export_settings['table']
        operator_department_urls = operators_export_settings['department_urls']
        return cls(export_table, operator_department_urls)

    def update_staff_entries(self):
        existing_uids = {
            str(x) for x in iter(
                CallCenterStaffEntry.objects.using(cars.settings.DB_RO_ID)
                .select_related('user')
                .values_list('user__uid', flat=True)
            )
        }
        staff_uids = itertools.chain.from_iterable(
            self.staff_info_helper.get_staff_uids_from_specific_department(d)
            for d in self._operator_department_urls
        )
        existing_uids.update(staff_uids)

        uids_to_update = list(existing_uids)
        total_entries_count = total_new_entries_count = 0

        for offset in range(0, len(uids_to_update), self.MAX_UIDS_PER_REQUEST):
            request_uids = uids_to_update[offset:offset + self.MAX_UIDS_PER_REQUEST]

            entries_count, new_entries_count = self.staff_info_helper.update_staff_entries(request_uids)

            total_entries_count += entries_count
            total_new_entries_count += new_entries_count

        self._staff_stat_access_helper.update_stat_access_entries()

        return total_entries_count, total_new_entries_count

    def export_staff_entries(self, check_data_exists=True):
        if check_data_exists:
            today = timezone.now().date()

            if self._yt_helper.check_table_data_exists(today):
                return

        data_to_store = self._iter_data_to_store()
        self._yt_helper.export_data(data_to_store)

    def _iter_data_to_store(self):
        staff_entries = CallCenterStaffEntry.objects.using(cars.settings.DB_RO_ID).select_related('user')

        today = timezone.now().date().strftime('%Y-%m-%d')

        for entry in staff_entries:
            data = {
                'date': today,
                'uid': str(entry.user.id),
                'uid_staff': str(entry.user.uid),
                'username': entry.username,
                'full_name': entry.full_name,
                'work_phone': entry.work_phone,
                'mobile_phone': str(entry.mobile_phone),
                'email': str(entry.email),
                'department_url': entry.department_url,
                'yang_worker_id': entry.yang_worker_id,
                'quit_at': entry.quit_at.strftime('%Y-%m-%d') if entry.is_deleted or entry.is_dismissed else None
            }
            yield data
