import logging

from django.utils import timezone

from cars.core.saas_drive_admin import SaasDriveAdminClient

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.core.common_helper import YTHelper


LOGGER = logging.getLogger(__name__)


class TraitExportHelper(object):
    saas_admin_client = SaasDriveAdminClient.from_settings()

    def __init__(self, export_table, roles_to_export, *, page_limit_per_role=None, specific_fields=None):
        self._yt_helper = YTHelper(export_table)

        self._roles_to_export = roles_to_export
        self._page_limit_per_role = page_limit_per_role
        self._specific_fields = specific_fields

    @classmethod
    def from_cc_included_roles_settings(cls):
        roles_export_settings = settings['callcenter']['export']['roles']

        export_table = roles_export_settings['included_roles_table']
        roles_to_include = roles_export_settings['included_roles_list']
        page_limit_per_role = roles_export_settings['page_limit_per_role']

        return cls(export_table, roles_to_include, page_limit_per_role=page_limit_per_role)

    @classmethod
    def from_cc_excluded_roles_settings(cls):
        roles_export_settings = settings['callcenter']['export']['roles']

        export_table = roles_export_settings['excluded_roles_table']

        roles_to_exclude = roles_export_settings['excluded_roles_list']

        all_roles = cls.saas_admin_client.get_all_roles_list()
        excluded_role_prefixes = tuple(role.rstrip('*') for role in roles_to_exclude)
        roles_to_include = [role for role in all_roles if not role.startswith(excluded_role_prefixes)]

        page_limit_per_role = roles_export_settings['page_limit_per_role']

        return cls(export_table, roles_to_include, page_limit_per_role=page_limit_per_role)

    @classmethod
    def from_cc_user_available_info_settings(cls):
        export_settings = settings['callcenter']['export']['support_user_available_info']

        export_table = export_settings['table']

        actions_to_include = export_settings['actions']
        roles_to_include = cls.saas_admin_client.get_all_roles_with_actions(actions_to_include)

        page_limit_per_role = export_settings['page_limit_per_role']
        specific_fields = export_settings['specific_fields']

        return cls(export_table, roles_to_include, page_limit_per_role=page_limit_per_role, specific_fields=specific_fields)

    def export_user_roles(self, *, specific_fields=None, check_data_exists=True, append=True):
        LOGGER.info(
            'exporting users to {} with roles {}'
            .format(self._yt_helper.table_path, self._roles_to_export)
        )

        if check_data_exists:
            today = timezone.now().date()

            if self._yt_helper.check_table_data_exists(today):
                LOGGER.info('data dated as {} already exists'.format(today))
                return

        specific_fields = specific_fields or self._specific_fields

        data_to_store = self._iter_data_to_store(specific_fields)
        self._yt_helper.export_data(data_to_store, append=append)

    def _iter_data_to_store(self, specific_fields=None):
        all_user_data = {}  # pairs (id: data)

        today = timezone.now().date().strftime('%Y-%m-%d')

        for role in self._roles_to_export:
            for entry in self._collect_role_user_data(role, today):
                entry_uid = entry['uid']
                if entry_uid not in all_user_data:
                    all_user_data[entry_uid] = entry
                else:
                    all_user_data[entry_uid]['roles'].append(role)

        LOGGER.info('collected {} users to export'.format(len(all_user_data)))

        if specific_fields is None:
            yield from all_user_data.values()
        else:
            LOGGER.info('only fields {} are to be stored'.format(specific_fields))

            specific_fields = set(specific_fields)
            for entry in all_user_data.values():
                entry = {k: v for k, v in entry.items() if k in specific_fields}
                yield entry

    def _collect_role_user_data(self, role, date):
        user_entries = self.saas_admin_client.get_all_users_with_roles_list(
            role, total_pages_limit=self._page_limit_per_role
        )

        for entry in user_entries:
            full_name = ' '.join(filter(None, (entry['last_name'], entry['first_name'], entry['pn'])))

            data = {
                'date': date,
                'uid': entry['id'],
                'uid_staff': None,
                'username': entry['username'],
                'full_name': full_name,
                'work_phone': None,
                'mobile_phone': entry['setup']['phone']['number'],
                'email': entry['setup']['email']['address'],
                'roles': [role],
            }
            yield data
