import collections
import logging
import uuid

from django.db import transaction

from cars.callcenter.models import CallCenterStaffEntry

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.core.common_helper import YTHelper


LOGGER = logging.getLogger(__name__)


class YangWorkersInfoHelper(object):
    YangWorkerEntry = collections.namedtuple(
        'YangWorkerEntry', ('date', 'login', 'workerId')
    )

    MAX_ENTRIES_BATCH = 5000
    TOTAL_ENTRIES_BATCH = 5000  # get only last N items

    def __init__(self, table_path):
        self._yt_helper = YTHelper(table_path)

    @classmethod
    def from_settings(cls):
        table_path = settings['callcenter']['yang_workers_table']
        return cls(table_path)

    def update_staff_entries(self):
        info_update_entries = self._get_workers_info()

        with transaction.atomic(savepoint=False):
            entries_to_process = (
                CallCenterStaffEntry.objects.select_for_update()
                .filter(yang_worker_id__isnull=True)
            )

            entries_to_update = (e for e in entries_to_process if e.username in info_update_entries)

            for e in entries_to_update:
                info_update = info_update_entries[e.username]
                if info_update.workerId:
                    e.yang_worker_id = str(uuid.UUID(info_update.workerId))
                    e.save()

    def _get_workers_info(self):
        total_row_count = self._yt_helper.get_table_row_count()

        start_offset = total_row_count - self.TOTAL_ENTRIES_BATCH

        entries = {}  # login: entry
        min_date, max_date = None, None

        for start_index in range(start_offset, total_row_count, self.MAX_ENTRIES_BATCH):
            end_index = min(start_index + self.MAX_ENTRIES_BATCH, total_row_count)

            for raw_entry in self._yt_helper.import_data(start_index, end_index):
                entry = self._make_yang_entry(raw_entry)
                entries[entry.login] = entry

                min_date = min(min_date or entry.date, entry.date)  # compare dates as str
                max_date = max(max_date or entry.date, entry.date)

        if min_date == max_date:
            LOGGER.warning('some entries are supposed to be not processed: total count - {}'.format(total_row_count))

        return entries

    def _make_yang_entry(self, raw_entry):
        flattened_entry = (raw_entry[name] for name in self.YangWorkerEntry._fields)
        yang_entry = self.YangWorkerEntry._make(flattened_entry)
        return yang_entry
