import json
import logging

from multiprocessing.dummy import Pool as ThreadPool

from cars.core.util import datetime_helper, phone_number_helper

from cars.request_aggregator.core.phone_binding_helper import PhoneBindingHelper
from cars.request_aggregator.models.internal_cc_stats import CallCenterEntry, InternalCCVerb

from .api_helper import IncomingTelephonyApiHelper
from .syncing_helper import IncomingTelephonyRequestTimeSyncHelper

from ..base.collecting_helper import CollectingHelperBase


LOGGER = logging.getLogger(__name__)


class IncomingTelephonyCollectingHelper(CollectingHelperBase):
    phone_binding_helper = PhoneBindingHelper()

    @classmethod
    def from_settings(cls):
        return cls(
            api_helper=IncomingTelephonyApiHelper.from_settings(),
            time_sync_helper=IncomingTelephonyRequestTimeSyncHelper.from_settings(),
        )

    def _filter_already_stored_data(self, raw_entries, start_timestamp, end_timestamp):
        # uniqueness is not guaranteed, some kind of id is required

        time_range = tuple(datetime_helper.timestamp_to_datetime(t) for t in (start_timestamp, end_timestamp))

        # warning: call id may be REALTIME (ADDMEMBER or REMOVEMEMBER verbs) that produces large output,
        #   further id comparison is strongly recommended
        already_stored_entries = {
            (r.call_id, r.time_id, r.verb) for r in
            CallCenterEntry.objects.filter(
                call_id__in={e.call_id for e in raw_entries},
                time_id__range=time_range,
            )
        }
        raw_entries_to_store = (
            item for item in raw_entries
            if (item.call_id, item.time_id, item.verb) not in already_stored_entries
        )
        return raw_entries_to_store

    def _process_entry(self, raw_entry):
        meta_info = {}

        time_id = raw_entry.time_id

        verb = raw_entry.verb
        verb_instance = InternalCCVerb.try_make(verb)

        if verb_instance.has_phone():
            phone = phone_number_helper.normalize_phone_number(raw_entry.data)
            if phone is None:
                meta_info['original_phone'] = raw_entry.data
        else:
            phone = None

        if phone:
            # temporarily disable due to multiple direct user bindings
            # user_binding, _ = self._phone_binding_helper.try_bind_phone(phone)
            user_binding = None
        else:
            user_binding = None

        queue_name = raw_entry.queue
        call_id = raw_entry.call_id

        if verb_instance.has_agent():
            agent = raw_entry.agent

            agent_work_phone = self.staff_info_helper.try_extract_agent_work_phone(agent)
            staff_entry = self.staff_info_helper.get_agent_entry(work_phone=agent_work_phone)

        else:
            agent = None
            staff_entry = None

        entry = CallCenterEntry(
            time_id=time_id, phone=phone, queue_name=queue_name,
            verb=verb, call_id=call_id, agent=agent,
            user_binding=user_binding, staff_entry_binding=staff_entry,
            meta_info=(meta_info or None),
        )
        return entry

    def _apply_entries(self, entries):
        CallCenterEntry.objects.bulk_create(entries)
