import datetime
import enum
import itertools
import logging

from django.db.models import Q

from cars.core.telephony import TelephonyQueue
from cars.core.util import datetime_helper

import cars.settings

from cars.request_aggregator.core.common_helper import collection_to_mapping
from cars.request_aggregator.models.internal_cc_stats import CallCenterEntry, InternalCCVerb
from cars.request_aggregator.models.call_tags import RequestOriginType
from cars.request_aggregator.serializers.common import AggregatedResponseEntry

from ..base.processing_helper import TelephonyProcessingHelperBase


LOGGER = logging.getLogger(__name__)


class IncomingCallStatus(enum.Enum):
    PENDING = 'pending'
    CONNECTED = 'connected'
    SERVICED = 'serviced'
    NOT_SERVICED = 'not_serviced'


class IncomingTelephonyProcessingHelper(TelephonyProcessingHelperBase):
    CALL_TIMEOUT = datetime.timedelta(hours=1)

    def get_requests(self, request=None, limit=None, **kwargs):
        if self._check_phone_not_matched(**kwargs):
            return []

        enter_entry_filters, connect_entry_filters = self._make_call_filters(**kwargs)

        if not (enter_entry_filters or connect_entry_filters) and limit is None:
            raise Exception('at least one filter must be provided')

        call_ids = self._get_call_ids(enter_entry_filters, connect_entry_filters, limit)

        # ordering does not guarantee correct ordering
        raw_call_entries = (
            CallCenterEntry.objects.using(cars.settings.DB_RO_ID).filter(call_id__in=call_ids).order_by('-time_id')
        )

        entry_tags_mapping = self.tag_description_helper.get_call_tags(
            (e.id for e in raw_call_entries),
            (
                    Q(request_origin=RequestOriginType.CARSHARING.value) |
                    Q(request_origin=RequestOriginType.CARSHARING_VIP.value)
            )
        )

        users_mapping = self._make_call_users_mapping(raw_call_entries, request, kwargs)

        grouped_call_entries = collection_to_mapping(raw_call_entries, attr_key='call_id')

        access_mapping = self._make_access_mapping(request)

        call_entries = self._transform_raw_entries(
            grouped_call_entries, entry_tags_mapping, users_mapping, access_mapping, limit
        )

        return call_entries

    def _get_call_ids(self, enter_entry_filters, connect_entry_filters, limit):
        enter_entry_call_ids_qs = (
            CallCenterEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(
                verb=InternalCCVerb.ENTER_QUEUE.value,
                **enter_entry_filters
            )
            .values_list('call_id', flat=True)
        )

        connect_entry_call_ids_qs = (
            CallCenterEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(
                verb=InternalCCVerb.CONNECT.value,
                **connect_entry_filters
            )
            .values_list('call_id', flat=True)
        )

        if enter_entry_filters and connect_entry_filters:
            call_ids = enter_entry_call_ids_qs.filter(call_id__in=list(connect_entry_call_ids_qs))
        elif connect_entry_filters:
            call_ids = connect_entry_call_ids_qs
        else:  # only enter filters or no filters at all
            call_ids = enter_entry_call_ids_qs

        call_ids = list(call_ids.order_by('-time_id')[:limit])
        return call_ids

    def _check_phone_not_matched(self, user=None, phone_number=None, **kwargs):
        # user phone number is not specified neither explicitly nor present in user info
        return user is not None and phone_number is None

    def _make_call_filters(self, phone_number=None, since=None, until=None, staff_entry_binding=None, **kwargs):
        # as phone number and time_enter are specified in ENTERQUEUE entry only, filters must be split on 2 parts
        enter_entry_filters, connect_entry_filters = {}, {}

        if phone_number is not None:
            enter_entry_filters['phone'] = phone_number

        if since is not None:
            enter_entry_filters['time_id__gte'] = since

        if until is not None:
            enter_entry_filters['time_id__lt'] = until

        if staff_entry_binding is not None:
            connect_entry_filters['staff_entry_binding'] = staff_entry_binding

            if since is not None:
                connect_entry_filters['time_id__gte'] = since

            if until is not None:
                connect_entry_filters['time_id__lt'] = until + self.CALL_TIMEOUT

        return enter_entry_filters, connect_entry_filters

    def _transform_raw_entries(self, grouped_call_entries, entry_tags_mapping, users_mapping, access_mapping, limit):
        response_entries = []

        entry_limit = range(limit) if limit is not None else itertools.count()

        for _, (call_id, entry_parts) in zip(entry_limit, grouped_call_entries.items()):
            entry_parts.sort(key=lambda x: -x.verb_instance.precedence, reverse=True)

            first_entry_part = entry_parts[0]

            if not first_entry_part.verb_instance.is_enter_entry():
                LOGGER.error('incorrect order of entry parts for call id {}'.format(first_entry_part.call_id))
                continue

            processed_entry = self._compile_entries(entry_parts, entry_tags_mapping, users_mapping, access_mapping)

            if processed_entry is not None:
                processed_entry_as_mapping = processed_entry._asdict()
                response_entries.append(processed_entry_as_mapping)

        return response_entries

    def _compile_entries(self, entry_parts, entry_tags_mapping, users_mapping, access_mapping):
        response_entry = None

        for entry in entry_parts:
            entry_time = entry.time_id.timestamp()
            entry_verb = entry.verb_instance

            if entry_verb.is_enter_entry():
                if response_entry is not None:
                    # entry is duplicated; to be removed using unique external id restriction
                    tags = entry_tags_mapping.get(str(entry.id), [])
                    response_entry.tags.extend(tags)
                    continue

                phone = str(entry.phone) if entry.phone is not None else entry.phone

                if access_mapping['has_access_to_call_id']:
                    data_url = self.get_call_url(entry.call_id)
                else:
                    data_url = None

                data = {'track_url': data_url, 'terminated_by': 'caller'}

                tags = entry_tags_mapping.get(str(entry.id), [])

                if entry.queue_name == TelephonyQueue.CARSHARING_VIP.value:
                    request_origin = RequestOriginType.CARSHARING_VIP.name
                else:
                    request_origin = RequestOriginType.CARSHARING.name

                request_description = {
                    'id': entry.id,
                    'origin': request_origin,
                    'queue': entry.queue_name,
                    'type': self.REQUEST_TYPE.value,
                }

                agent = self.staff_info_helper.format_staff_entry(None)

                user = users_mapping.get(entry.id, None)

                response_entry = AggregatedResponseEntry(
                    time_enter=entry_time, time_exit=None,
                    duration=None, duration_print='',
                    connect_trial_count=0,  time_connect=None,
                    status=IncomingCallStatus.PENDING.value,
                    phone=phone, operators=[agent], tags=tags,
                    request=request_description, user=user,
                    message=None, data_url=data_url, data=data,
                )

            elif entry_verb.is_decision_entry():
                agent_work_phone = self.staff_info_helper.try_extract_agent_work_phone(entry.agent)
                agent_info_entry = self.staff_info_helper.get_agent_entry(work_phone=agent_work_phone)
                agent = self.staff_info_helper.format_staff_entry(agent_info_entry)

                connect_trial_count = response_entry.connect_trial_count
                operators = response_entry.operators

                if connect_trial_count > 0:
                    operators.append(agent)
                else:
                    operators[-1] = agent  # replace mock

                if entry_verb.is_connect_entry():
                    kwargs = {'time_connect': entry_time, 'status': IncomingCallStatus.CONNECTED.value}
                else:
                    kwargs = {}

                response_entry = response_entry._replace(connect_trial_count=connect_trial_count + 1, **kwargs)

            elif entry_verb.is_abandon_entry():
                response_entry = response_entry._replace(
                    time_exit=entry_time, status=IncomingCallStatus.NOT_SERVICED.value
                )

            elif entry_verb.is_completion_entry() or entry_verb.is_transfer_entry():
                # time exit and time connect may have the same time and
                #  they may follow in incorrect order - no reason to sort correctly
                time_connect = response_entry.time_connect or entry_time
                duration = entry_time - time_connect

                if entry_verb != InternalCCVerb.COMPLETE_CALLER:
                    response_entry.data['terminated_by'] = 'agent'

                response_entry = response_entry._replace(
                    time_exit=entry_time,
                    duration=duration,
                    duration_print=datetime_helper.duration_to_str(duration),
                    status=IncomingCallStatus.SERVICED.value,
                )

            else:
                # just skip unknown entries like call transfer
                LOGGER.warning('entry {} has not been processed'.format(entry))

        if response_entry and not access_mapping['has_access_to_operators']:
            response_entry.operators.clear()

        return response_entry
