import logging

from django.db import transaction

from cars.core.util import phone_number_helper, datetime_helper
from cars.callcenter.core import StaffInfoHelper

from cars.request_aggregator.core.phone_binding_helper import PhoneBindingHelper
from cars.request_aggregator.models.internal_cc_stats import CallCenterEntry, InternalCCVerb
from cars.request_aggregator.models.call_center_common import CallStatSyncStatus, SyncOrigin


LOGGER = logging.getLogger(__name__)


class IncomingTelephonyServiceHelper(object):
    MAX_ENTRIES_LIMIT = 1000

    phone_binding_helper = PhoneBindingHelper()
    staff_info_helper = StaffInfoHelper.make_default()

    @classmethod
    def from_settings(cls):
        return cls()

    def update_incoming_bad_formatted_phone_numbers(self):
        with transaction.atomic(savepoint=False):
            bad_phone_number_entries = (
                CallCenterEntry.objects.select_for_update()
                .filter(phone__startswith='+98')
                [:self.MAX_ENTRIES_LIMIT]
            )

            for entry in bad_phone_number_entries:
                origin_phone = str(entry.phone).lstrip('+')
                normalized_phone = phone_number_helper.normalize_phone_number(origin_phone)

                LOGGER.info('update number "{}" to "{}"'.format(origin_phone, normalized_phone))

                entry.phone = normalized_phone
                entry.save()

        total_numbers = len(bad_phone_number_entries)
        LOGGER.info('total numbers updated: {}'.format(total_numbers))

        return total_numbers

    def update_user_bindings(self, offset, count):
        with transaction.atomic(savepoint=False):
            entries = (
                CallCenterEntry.objects.select_for_update()
                .filter(
                    verb__in=(InternalCCVerb.COMPLETE_CALLER.value, InternalCCVerb.COMPLETE_AGENT.value,)
                )
                .order_by('id')
                [offset:offset + count]
            )

            for entry in entries:
                if entry.user_binding is None and entry.phone:
                    user_binding, _ = self.phone_binding_helper.try_bind_phone(entry.phone)
                    entry.user_binding = user_binding
                    entry.save()

        total_numbers = len(entries)
        LOGGER.info('total numbers updated: {}'.format(total_numbers))

        return total_numbers

    def update_incoming_staff_bindings(self, offset, count):
        limit = min(count, self.MAX_ENTRIES_LIMIT)

        performed = -1
        updated = 0

        with transaction.atomic(savepoint=False):
            entries = (
                CallCenterEntry.objects.select_for_update()
                .filter(
                    staff_entry_binding__isnull=True,
                    agent__isnull=False,
                )
                .order_by('id')
                [offset:offset + limit]
            )

            for performed, entry in enumerate(entries):
                agent_work_phone = self.staff_info_helper.try_extract_agent_work_phone(entry.agent)
                agent_entry = self.staff_info_helper.get_agent_entry(work_phone=agent_work_phone)
                if agent_entry is not None:
                    entry.staff_entry_binding = agent_entry
                    entry.save()
                    updated += 1

            values_to_update = {
                'last_data_sync_time': datetime_helper.utc_now(),
            }

            sync_entry = CallStatSyncStatus.objects.filter(
                origin=SyncOrigin.SERVICE_CC_INTERNAL_IN_UPDATE_STAFF_BINDING.value,
            ).first()

            if sync_entry is None:
                sync_entry = CallStatSyncStatus.objects.create(
                    origin=SyncOrigin.SERVICE_CC_INTERNAL_IN_UPDATE_STAFF_BINDING.value, **values_to_update
                )
            else:
                for key, value in values_to_update.items():
                    setattr(sync_entry, key, value)
                sync_entry.save()

        return (performed + 1), updated
