from django.db.models import Q

import cars.settings

from cars.core.telephony import TelephonyQueue
from cars.request_aggregator.core.request_tags import RequestTagBindingHelper

from cars.request_aggregator.models.internal_cc_stats import CallCenterEntry, InternalCCVerb
from cars.request_aggregator.models.call_center_common import SyncOrigin
from cars.request_aggregator.models.call_tags import RequestOriginType


class InternalCCIncomingTagBindingHelperBase(RequestTagBindingHelper):
    def _get_internal_cc_entry_to_process_values(self, since, until, filter_args):
        filter_kwargs = {
            'time_id__range': (since, until),
            'verb': InternalCCVerb.ENTER_QUEUE.value,
        }

        enter_entries = (
            CallCenterEntry.objects
            .using(cars.settings.DB_RO_ID)
            .filter(*filter_args, **filter_kwargs)
            .values('id', 'phone', 'call_id')
        )

        enter_entries_mapping = {e['call_id']: e for e in enter_entries}

        connect_entries = (
            CallCenterEntry.objects
            .using(cars.settings.DB_RO_ID)
            .select_related('staff_entry_binding')
            .filter(
                call_id__in=enter_entries_mapping,
                verb=InternalCCVerb.CONNECT.value,
                staff_entry_binding__isnull=False,
            )
            .values('call_id', 'staff_entry_binding__user_id')
        )

        entries_to_process = [dict(e, **enter_entries_mapping[e['call_id']]) for e in connect_entries]
        return entries_to_process


class InternalCCIncomingGenericTagBindingHelper(InternalCCIncomingTagBindingHelperBase):
    @classmethod
    def from_settings(cls):
        return cls(
            stat_sync_origin=SyncOrigin.CC_INTERNAL_CARSHARING_TAGS_BINDING,
            request_origin=RequestOriginType.CARSHARING
        )

    def _get_entry_to_process_values(self, since, until):
        filter_argument = ~Q(queue_name=TelephonyQueue.CARSHARING_VIP.value)
        return self._get_internal_cc_entry_to_process_values(since, until, (filter_argument, ))


class InternalCCIncomingVipTagBindingHelper(InternalCCIncomingTagBindingHelperBase):
    @classmethod
    def from_settings(cls):
        return cls(
            stat_sync_origin=SyncOrigin.CC_INTERNAL_CARSHARING_VIP_TAGS_BINDING,
            request_origin=RequestOriginType.CARSHARING_VIP
        )

    def _get_entry_to_process_values(self, since, until):
        filter_argument = Q(queue_name=TelephonyQueue.CARSHARING_VIP.value)
        return self._get_internal_cc_entry_to_process_values(since, until, (filter_argument, ))
