import collections
import datetime
import json
import logging

import cars.settings

from cars.core.solomon import SolomonHelper
from cars.callcenter.core.settings_manager import LoadBalanceHelper
from cars.request_aggregator.core.request_time_sync_helper import RequestStateSyncHelper

from cars.request_aggregator.models.call_center_common import SyncOrigin


LOGGER = logging.getLogger(__name__)


class LoadBalanceSyncHelper(object):
    DEFAULT_SYNC_PERIOD = datetime.timedelta(hours=1)

    solomon_helper = SolomonHelper('request_aggregator', 'callcenter.load_balance')

    def __init__(self):
        self._helper = LoadBalanceHelper.from_settings()
        self._sync_helper = RequestStateSyncHelper(
            SyncOrigin.CC_LOAD_BALANCE,
            default_state={'sync_period': self.DEFAULT_SYNC_PERIOD.total_seconds()}
        )

    @classmethod
    def from_settings(cls):
        return cls()

    def _trunc_timestamp(self, timestamp, sync_period):
        if timestamp is None:
            return None

        sync_period_s = sync_period.total_seconds()
        return int(timestamp // sync_period_s * sync_period_s)

    def process(self):
        now_ts = datetime.datetime.utcnow().timestamp()

        counters = collections.Counter()

        for state in self._sync_helper.iter_states_to_process():
            sync_period = datetime.timedelta(seconds=state['sync_period'])
            truncated_now_ts = self._trunc_timestamp(now_ts, sync_period)

            distribute_apps_settings = cars.settings.CALLCENTER['distribute_apps']

            for city, application_source in distribute_apps_settings.items():
                state.setdefault(application_source, {})

                last_sync_ts = state[application_source].get('last_sync_timestamp', None)
                truncated_last_sync_ts = self._trunc_timestamp(last_sync_ts, sync_period)

                if last_sync_ts is None or truncated_now_ts > truncated_last_sync_ts:
                    try:
                        current_interval_settings = self._helper.update_settings_from_calendar(application_source)
                    except Exception as exc:
                        LOGGER.exception('error updating load balance settings of {}'.format(application_source))
                        is_successful = False
                    else:
                        LOGGER.info(
                            'successfully updated load balance settings of {} to {}'
                            .format(application_source, json.dumps(current_interval_settings))
                        )
                        is_successful = True

                    state[application_source]['last_sync_timestamp'] = now_ts
                    state[application_source]['is_successful'] = is_successful

                    counter_name = 'success' if is_successful else 'fail'
                    counters[counter_name] += 1

        self.solomon_helper.increment_counters(**counters)
