import collections
import concurrent.futures
import datetime
import logging

from cars.settings import REQUEST_AGGREGATOR as settings

from cars.core.telephony import TelephonyHelperBase, OutgoingTelephonyApiHelper as OutgoingTelephonyApiHelperBase
from cars.core.util import datetime_helper, phone_number_helper
from cars.callcenter.core import StaffInfoHelper
from cars.callcenter.models import CallCenterStaffStatAccessEntry

from cars.request_aggregator.core.common_helper import RateLimiter
from cars.request_aggregator.models.internal_cc_stats import InternalCCAltayCallType


LOGGER = logging.getLogger(__name__)


class OutgoingTelephonyApiHelper(OutgoingTelephonyApiHelperBase):
    def _get_internal_phone_numbers_to_collect(self):
        raw_work_phones = (
            CallCenterStaffStatAccessEntry.objects.select_related()
            .filter(
                outgoing_stat_granted_at__isnull=False,
                outgoing_stat_refused_at__isnull=True,
            )
            .values_list('staff_entry__work_phone', flat=True)
        )

        work_phones = [str(phone) for phone in raw_work_phones if phone]
        return work_phones


class AltayApiHelper(TelephonyHelperBase):
    DEFAULT_LIMIT = 20
    MAX_LIMIT = 3000

    MAX_WORKERS_COUNT = MAX_RATE_LIMIT = 15

    staff_info_helper = StaffInfoHelper.make_default()

    API_TEMPLATE = (
        'https://altay-callcenter.common-ext.yandex-team.ru/v1/callcenter/calls'
        '?user={robot_uid}&author_id={uid}&offset={offset}&limit={limit}'
    )

    API_RESPONSE = collections.namedtuple(
        'API_RESPONSE', [
            'external_id', 'time_enter', 'time_connect', 'time_exit', 'duration', 'wait_duration',
            'status', 'noc_status', 'call_type', 'comment', 'phone', 'staff_entry_binding', 'has_record', 'meta_info',
        ]
    )

    def __init__(self, robot_uid, request_timeout=15):
        super().__init__(
            request_timeout=request_timeout,
            pool_connections=self.MAX_WORKERS_COUNT,
            pool_maxsize=self.MAX_WORKERS_COUNT
        )
        self._robot_uid = robot_uid
        self._rate_limiter = RateLimiter(self.MAX_RATE_LIMIT)

    @classmethod
    def from_settings(cls):
        robot_uid = settings['altay']['robot_uid']
        request_timeout = settings['altay']['request_timeout']
        return cls(robot_uid=robot_uid, request_timeout=request_timeout)

    def get_update(self, uid, offset=0, limit=DEFAULT_LIMIT, *, re_raise=True):
        if isinstance(uid, collections.Iterable):
            return self._get_update_async(uid, offset, limit, re_raise=re_raise)

        limit = min(limit, self.MAX_LIMIT)
        url = self.API_TEMPLATE.format(robot_uid=self._robot_uid, uid=uid, offset=offset, limit=limit)

        self._rate_limiter.wait()

        raw_items = self._perform_request(url, raise_for_status=re_raise)

        if raw_items is not None:
            items = [self._process_entry(raw_entry) for raw_entry in raw_items]
            data_update = {uid: items}
        else:
            data_update = {}

        return data_update

    def _get_update_async(self, uid_collection, offset, limit, *, re_raise=True):
        data_update = {}  # uid: items mapping

        with concurrent.futures.ThreadPoolExecutor(self.MAX_WORKERS_COUNT) as executor:
            futures = (
                executor.submit(self.get_update, uid, offset, limit, re_raise=re_raise)
                for uid in uid_collection
            )

            for future in concurrent.futures.as_completed(futures):
                try:
                    result = future.result()
                    data_update.update(result)
                except Exception as exc:
                    LOGGER.exception('future raised an exception: {}'.format(exc))
                    if re_raise:
                        raise

        return data_update

    def _perform_request(self, url, *, raise_for_status=True, **kwargs):
        data = None

        raw_data = super()._perform_request(url, verify=False, raise_for_status=raise_for_status)

        if raw_data is not None:
            data = raw_data['items']

        return data

    def _process_entry(self, raw_entry):
        # raw entry contains:
        #   'id', 'status', 'noc_status', 'call_time', 'duration', 'wait_duration', 'comment',
        #   'phone', 'call_type', 'has_record', 'user.id', 'user.login'
        meta_info = {}

        external_id = raw_entry['id']

        time_enter = datetime_helper.timestamp_ms_to_datetime(raw_entry['call_time'])

        time_exit = time_enter

        wait_duration = raw_entry.get('wait_duration', None)
        duration = raw_entry.get('duration', None)

        if wait_duration is not None:
            if wait_duration < 0:
                meta_info['wait_duration'] = wait_duration
                wait_duration = 0

            time_connect = time_enter + datetime.timedelta(seconds=wait_duration)
            time_exit += datetime.timedelta(seconds=wait_duration)
        else:
            time_connect = None

        if duration is not None:
            if duration < 0:
                meta_info['duration'] = duration
                duration = 0

            time_exit += datetime.timedelta(seconds=duration)

        status = raw_entry['status']
        noc_status = raw_entry.get('noc_status', None)

        comment = raw_entry.get('comment', '')

        raw_phone = raw_entry['phone']
        phone = phone_number_helper.normalize_phone_number(raw_phone)

        if phone is None:
            meta_info['phone'] = raw_phone

        staff_entry_binding = self.staff_info_helper.get_agent_entry(username=raw_entry['user']['login'])

        if staff_entry_binding is None:
            meta_info['user'] = raw_entry['user']

        try:
            call_type = InternalCCAltayCallType[raw_entry['call_type']].value  # convert using enum
        except (TypeError, ValueError):
            meta_info['call_type'] = raw_entry['call_type']
            call_type = None

        has_record = raw_entry['has_record']

        if 'task_id' in raw_entry:
            meta_info['task_id'] = raw_entry['task_id']

        if 'company_id' in raw_entry:
            meta_info['company_id'] = raw_entry['company_id']

        return self.API_RESPONSE(
            external_id=external_id,
            status=status,
            noc_status=noc_status,
            time_enter=time_enter,
            time_connect=time_connect,
            time_exit=time_exit,
            duration=duration,
            wait_duration=wait_duration,
            call_type=call_type,
            comment=comment,
            phone=phone,
            staff_entry_binding=staff_entry_binding,
            has_record=has_record,
            meta_info=(meta_info or None),
        )
