import collections
import functools
import itertools
import logging
import operator

from django.db import transaction
from django.db.models import Q

from cars.core.util import datetime_helper
from cars.callcenter.models import CallCenterStaffEntry
from cars.settings import REQUEST_AGGREGATOR as settings

from cars.request_aggregator.core.common_helper import collection_to_mapping
from cars.request_aggregator.models.internal_cc_stats import OutgoingCallCenterEntry, CallCenterAltayOutgoingEntry

from .api_helper import OutgoingTelephonyApiHelper, AltayApiHelper
from .syncing_helper import OutgoingTelephonyRequestTimeSyncHelper, CCInternalAltayOutSyncHelper

from ..base.collecting_helper import CollectingHelperBase


LOGGER = logging.getLogger(__name__)


# deprecated
class OutgoingTelephonyCollectingHelper(CollectingHelperBase):
    SENSOR_NAME = 'OUTGOING'

    @classmethod
    def from_settings(cls):
        return cls(
            api_helper=OutgoingTelephonyApiHelper.from_settings(),
            time_sync_helper=OutgoingTelephonyRequestTimeSyncHelper.from_settings(),
        )

    def update_data(self, batch_size=10):
        return super().update_data(batch_size)

    def _filter_already_stored_data(self, raw_entries, start_timestamp, end_timestamp):
        time_range = tuple(datetime_helper.timestamp_to_datetime(t) for t in (start_timestamp, end_timestamp))
        already_stored_entries = {
            (r.time_enter, r.time_exit, r.phone) for r in
            OutgoingCallCenterEntry.objects.filter(time_enter__range=time_range)
        }
        raw_entries_to_store = (
            item for item in raw_entries
            if (item.time_enter, item.time_exit, item.phone) not in already_stored_entries
        )
        return raw_entries_to_store

    def _handle_entry(self, raw_entry, entries_stat):
        entries_stat[self.SENSOR_NAME] += 1
        return True

    def _process_entry(self, raw_entry):
        agent_work_phone = raw_entry.agent
        staff_entry = self.staff_info_helper.get_agent_entry(work_phone=agent_work_phone)

        entry = OutgoingCallCenterEntry(
            time_enter=raw_entry.time_enter,
            time_connect=raw_entry.time_connect,
            time_exit=raw_entry.time_exit,
            duration=raw_entry.duration,
            phone=raw_entry.phone,
            agent=agent_work_phone,
            staff_entry_binding=staff_entry,
        )
        return entry

    def _apply_entries(self, entries):
        OutgoingCallCenterEntry.objects.bulk_create(entries)


class CCInternalAltayOutCollectingHelper(object):
    DEFAULT_SENSOR_NAME = 'OUTGOING_ALTAY'
    UPDATE_ENTRY_SENSOR_NAME = 'OUTGOING_ALTAY_UPDATE'

    MAX_EXTERNAL_ID_REQUEST_BATCH = 1000

    not_set = object()

    def __init__(self, api_helper, time_sync_helper, operator_uid_collection):
        assert isinstance(api_helper, AltayApiHelper)
        assert isinstance(time_sync_helper, CCInternalAltayOutSyncHelper)
        self._api_helper = api_helper
        self._sync_helper = time_sync_helper
        self._operator_uid_collection = operator_uid_collection

    @classmethod
    def from_settings(cls):
        departments = settings['callcenter']['export']['operators']['department_urls']
        operator_uid_collection = cls._get_operator_uid_collection(departments)
        return cls(
            api_helper=AltayApiHelper.from_settings(),
            time_sync_helper=CCInternalAltayOutSyncHelper.from_settings(),
            operator_uid_collection=operator_uid_collection,
        )

    @classmethod
    def _get_operator_uid_collection(cls, departments):
        department_filters = functools.reduce(operator.or_, (Q(department_url__startswith=x) for x in departments))
        user_uids = list(CallCenterStaffEntry.objects.filter(department_filters).values_list('user__uid', flat=True))
        return user_uids

    def update_data(self):
        all_entries_stat = collections.Counter()

        for uid, offset, limit in self._sync_helper.iter_states_to_process():
            entries_stat = collections.Counter()

            # do not fail if part of data was failed to update
            entries_mapping = self._api_helper.get_update(
                uid or self._operator_uid_collection, offset, limit, re_raise=False
            )

            with transaction.atomic(savepoint=False):
                existing_entries_info = self._get_existing_entries_info(entries_mapping)

                for specific_uid, raw_entries in entries_mapping.items():
                    if not raw_entries:
                        continue

                    filtered_entries = self._filter_already_stored_data(raw_entries, existing_entries_info)
                    entry_mappings, entry_update_mappings = self._process_entries(filtered_entries, entries_stat)

                    if not (entry_mappings or entry_update_mappings):
                        continue

                    self._apply_entries(entry_mappings, batch_size=10)
                    self._apply_entry_updates(entry_update_mappings)

                    if len(raw_entries) == limit:  # all batch obtained and more entries may exist
                        last_entry_id = raw_entries[-1].external_id

                        last_entry_id_to_create = entry_mappings[-1]['external_id'] if entry_mappings else None
                        last_entry_id_to_update = entry_update_mappings[-1]['external_id'] if entry_update_mappings else None

                        # if last entry requires update or it's to be created, further update may be required
                        if last_entry_id == last_entry_id_to_create or last_entry_id == last_entry_id_to_update:
                            self._sync_helper.add_extra_state(specific_uid, current_offset=offset, current_limit=limit)

            LOGGER.info('processed entries: {}'.format(entries_stat))
            all_entries_stat.update(entries_stat)

        return all_entries_stat

    def _filter_already_stored_data(self, raw_entries, existing_entries_info):
        for e in raw_entries:
            are_entries_equal = False
            already_exists = False

            if e.external_id in existing_entries_info:
                already_exists = True
                existing_info = existing_entries_info[e.external_id]
                are_entries_equal = all(existing_info.get(k, self.not_set) == getattr(e, k) for k in existing_info)

            if not are_entries_equal:
                yield e, already_exists

    def _get_existing_entries_info(self, entries_mapping):
        existing_entries_info = {}

        external_ids = (e.external_id for e in itertools.chain.from_iterable(entries_mapping.values()))

        while True:
            external_ids_batch = list(itertools.islice(external_ids, self.MAX_EXTERNAL_ID_REQUEST_BATCH))

            if not external_ids_batch:
                break

            existing_entries_batch_info = (
                CallCenterAltayOutgoingEntry.objects
                .filter(external_id__in=external_ids_batch)
                .values('external_id', 'status', 'noc_status', 'duration', 'wait_duration', 'comment', 'has_record')
            )

            existing_entries_info.update(
                collection_to_mapping(existing_entries_batch_info, item_key='external_id', multiple_values=False)
            )

        return existing_entries_info

    def _process_entries(self, raw_entries, entries_stat):
        entries, entry_updates = [], []

        for entry, already_exists in raw_entries:
            c = entry_updates if already_exists else entries

            if self._handle_entry(entry, already_exists, entries_stat):
                c.append(self._process_entry(entry))

        return entries, entry_updates

    def _handle_entry(self, raw_entry, already_exists, entries_stat):
        sensor_name = self.DEFAULT_SENSOR_NAME if not already_exists else self.UPDATE_ENTRY_SENSOR_NAME
        entries_stat[sensor_name] += 1
        return True

    def _process_entry(self, raw_entry):
        assert isinstance(raw_entry, AltayApiHelper.API_RESPONSE)
        return raw_entry._asdict()

    def _apply_entries(self, entries, batch_size=None):
        instances = (CallCenterAltayOutgoingEntry(**e) for e in entries)

        batch = None
        try:
            if batch_size is None:
                batch = instances
                CallCenterAltayOutgoingEntry.objects.bulk_create(batch)
            else:
                instances = list(instances)
                for offset in range(0, len(instances), batch_size):
                    batch = instances[offset:offset + batch_size]
                    CallCenterAltayOutgoingEntry.objects.bulk_create(batch)
        except Exception:
            self._handle_applying_error(batch)
            raise

    def _handle_applying_error(self, entries):
        entries = list(entries)
        if len(entries) > 10:
            LOGGER.error('Error applying entries: one of {} items in batch is incorrect'.format(len(entries)))
        else:
            LOGGER.error('Error applying entries: {}'.format('; '.join(str(entry) for entry in entries)))

    def _apply_entry_updates(self, entry_updates):
        # atomicity is provided in the outer function
        for entry_update in entry_updates:
            existing_entry = (
                CallCenterAltayOutgoingEntry.objects.select_for_update()
                .get(external_id=entry_update['external_id'])
            )

            for key, value in entry_update.items():
                setattr(existing_entry, key, value)

            existing_entry.save()
