import datetime
import json
import logging

from cars.core.util import datetime_helper
from cars.request_aggregator.core.data_exporting_helper import DataExportingHelper, DataExportingSyncHelper

from cars.request_aggregator.models.internal_cc_stats import CallCenterAltayOutgoingEntry

LOGGER = logging.getLogger(__name__)


class InternalCCOutgoingDataExportingSyncHelper(DataExportingSyncHelper):
    entry_model_cls = CallCenterAltayOutgoingEntry
    ENTRY_TIME_LAG = datetime.timedelta(hours=12)

    @classmethod
    def from_settings(cls, sync_status_origin):
        return cls(
            stat_sync_origin=sync_status_origin,
            default_state={'next_entry_id': 1}
        )

    def _get_qs(self):
        return super()._get_qs().filter(time_enter__lte=datetime_helper.utc_now() - self.ENTRY_TIME_LAG)


class InternalCCOutgoingDataExportingHelper(DataExportingHelper):
    _sync_helper_cls = InternalCCOutgoingDataExportingSyncHelper

    def _make_export_entry(self, entry):
        data = {
            'id': str(entry.id),
            'agentUsername': self.get_agent_username(entry),
            'metaInfo': json.dumps(entry.meta_info) if entry.meta_info else None,
            'phone': self.get_original_phone(entry),
            'callId': entry.external_id,

            'callType': entry.call_type,
            'comment': entry.comment,
            'duration': entry.duration,
            'hasRecord': entry.has_record,
            'nocStatus': entry.noc_status,
            'status': entry.status,
            'timeEnter': self.time_to_repr(entry.time_enter),
            'timeConnect': self.time_to_repr(entry.time_connect),
            'timeExit': self.time_to_repr(entry.time_exit),
            'waitDuration': entry.wait_duration,
        }
        return data
